! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     PID controller
! *
! * PURPOSE:        PID controller
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. y       : controlled variable (sensor signal)                (-)
! *  2. w       : setpoint                                           (-)
! *
! * OUTPUTS
! * =======
! *  1. u       : control signal                                     (-)
! *
! * PARAMETERS
! * ==========
! *  1. propb   : proportional band                                  (-)
! *  2. tint    : integral time                                      (s)
! *  3. tdt     : derivative time                                    (s)
! *  4. dband   : deadband                                           (-)
! *  5. closloop: control mode (0=open loop, 1=closed loop)          (-)
! *  6. uman    : open loop control signal                           (-)
! *  7. umin    : lower limit on control signal                      (-)
! *  8. umax    : upper limit on control signal                      (-)
! *  9. tsamp   : sampling interval                                  (s)
! * 10. nfile   : controller number (parameters in file contN.par)   (-)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call
! *  2. nsample : sample number of previous controller execution
! *  3. nsample : sample number of previous controller sample
! *  4. intp    : integral term from previous call
! *  5. intp    : integral term from previous sample
! *  6. difp    : derivative term from previous call
! *  7. difp    : derivative term from previous sample
! *  8. errp    : error from previvous call
! *  9. errp    : error from previvous sample
! * 10. pidoutp : output from previous call
! * 11. pidoutp : output from previous sample
! * 12-19.      : (PAR(1)-PAR(8) read from file
! * 20. up      : controller output from previous sample
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 22, 1995
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   RETOLOG, PIDCONT
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           ASHRAE 825-RP Final Report
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type441(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=11,ni=2,no=1,np=10,&
                                             nfp=8,ns=nsv+nfp+1
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

! **** Declaration of local variables
        real         :: intp
        logical      :: retolog,closloop
        real         :: y,w,propb,tint,tdt,dband,uman,umin,umax,tsamp,&
                        difp,errp,up,wdb,pidcont,ux
        integer      :: nfile,is,nsample,i
        integer      :: itype=441

! **** Read in inputs
        y         = xin(1)
        w         = xin(2)
! **** Read in parameters
        propb     = par_v(1)
        tint      = par_v(2)
        tdt       = par_v(3)
        dband     = par_v(4)
        closloop  = retolog(par_v(5))
        uman      = par_v(6)
        umin      = par_v(7)
        umax      = par_v(8)
        tsamp     = par_v(9)
        nfile     = par_v(10)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
                do is = 4,(nsv-1),2
                    saved_v(is) = 0.0
                enddo   
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
            if (nfile>0) then
! **** First call of time-step - read parameters from file and store
                if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
                endif
! **** Get parameters that were read from file and then stored
                propb    = saved_v(nsv+1)
                tint     = saved_v(nsv+2)
                tdt      = saved_v(nsv+3)
                dband    = saved_v(nsv+4)
                closloop = retolog(saved_v(nsv+5))
                uman     = saved_v(nsv+6)
                umin     = saved_v(nsv+7)
                umax     = saved_v(nsv+8)
            endif
! **** Run controller if mode is closed loop else use "manual" value
            if (closloop) then
! **** Closed loop mode
                if (time>saved_v(1)) then
! **** First call of timestep
! **** Update previous sample instant values
                    do is=4,(nsv-1),2
                        saved_v(is+1)=saved_v(is)
                    enddo
                endif
! **** Update previous values
                intp = saved_v(5)
                difp = saved_v(7)
                errp = saved_v(9)
                up   = saved_v(11)
! **** Apply deadband around set-point
                call deadband(y,w,dband,wdb)
! **** Pid controller
                ux = pidcont(y,wdb,propb,tint,tdt,intp,difp,&
                               up,errp,tsamp,umax,umin)
! **** Save provisional values to be used in updating at next sample instant
                saved_v(4)  = intp
                saved_v(6)  = difp
                saved_v(8)  = errp
                saved_v(10) = ux
! **** Open loop mode
            else
                ux = uman
            endif
! **** Save current sample number and output
            saved_v(2)  = float(nsample)
            saved_v(20) = ux
        else
! **** Not a sample instant, set output to value from previous sample
! **** instant
            ux = saved_v(20)
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = ux
! **** Disallow freezing
        do i=1,no
            iostat(i) = 0
        enddo
! **** Return
        return
        end subroutine type441
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     E51 supply fan control
! *
! * PURPOSE:        Control supply static pressure by varying supply fan
! *                 speed. Override fan speed if static pressure exceeds
! *                 high limit.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. ps      : static pressure sensor                           (kPa)
! *  2. psset   : static pressure setpoint                         (kPa)
! *  3. sfstatus: supply fan status (1 = on, 0 = off)                (-)
! *  4. rfstatus: return fan status (1 = on, 0 = off)                (-)
! *
! * OUTPUTS
! * =======
! *  1. sfspd   : supply fan speed (0-1)                             (-)
! *
! * PARAMETERS
! * ==========
! *  1. propb   : proportional band                                (kPa)
! *  2. tint    : integral time                                      (s)
! *  3. tdt     : derivative time                                    (s)
! *  4. dband   : deadband                                         (kPa)
! *  5. propbhl : high limit override propotional band             (kPa)
! *  6. pshlimit: high limit override setpoint                     (kPa)
! *  7. closloop: control mode (0=open loop, 1=closed loop)          (-)
! *  8. sfspdman: open loop supply fan speed (0-1)                   (-)
! *  9. tsamp   : sampling interval                                  (s)
! * 10. nfile   : controller number (parameters in file contN.par)   (-)
!!!!!!!!!!!!!!!!!!!Drexel changed!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! * 11. location: 1-SA hot deck, 2-SA cold deck                      (-)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call
! *  2. nsample : sample number of previous controller execution
! *  3. nsample : sample number of previous controller sample
! *  4. intp    : integral term from previous call
! *  5. intp    : integral term from previous sample
! *  6. difp    : derivative term from previous call
! *  7. difp    : derivative term from previous sample
! *  8. errp    : error from previvous call
! *  9. errp    : error from previvous sample
! * 10. pidoutp : output from previous call
! * 11. pidoutp : output from previous sample
! * 12-19.      : (PAR(1)-PAR(8) read from file
! * 20. sfspd   : controller output from previous sample
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 22, 1995
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   RETOLOG,DEADBANDT,PIDCONT,SWITCH,SPAN,
!                        LOGICAND
!
!   REVISION HISTORY:    Shokouh Pourarian, Jin Wen
!
!   REFERENCE:           ASHRAE 825-RP Final Report
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type481(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=11,ni=4,no=1,np=11,&
                                             nfp=8,ns=nsv+nfp+1
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

 ! **** Declaration of local variables
        real         :: intp
        logical      :: retolog,sfstatus,rfstatus,closloop,auxdis,&
                        logicand
        real         :: ps,psset,propb,tint,tdt,dband,propbhl,pshlimit,&
                        sfspdman,tsamp,difp,errp,pidoutp,pssetdb,pidout,&
                        pidcont,contout,switch,dummy4,dummy3,dummy2,&
                        dummy1,sfspdhl,sfspd
        integer      :: nfile,is,nsample,i,location
        integer      :: itype=481

! **** Read in inputs
        ps        = xin(1)
        psset     = xin(2)
		sfstatus  = retolog(xin(3))
        rfstatus  = retolog(xin(4))
! **** Read in parameters
        propb     = par_v(1)
        tint      = par_v(2)
        tdt       = par_v(3)
        dband     = par_v(4)
        propbhl   = par_v(5)
        pshlimit  = par_v(6)
        closloop  = retolog(par_v(7))
        sfspdman  = par_v(8)
        tsamp     = par_v(9)
        nfile     = par_v(10)
		location  = nint(par_v(11))

! ---- Fault flag on
        if (location==1)then
			if (pshotset==0) then
				psset = psset
			elseif (pshotset==1) then
				psset = psset + Vpshotset
			endif
        elseif (location==2)then
			if (pscoldset==0) then
				psset = psset
			elseif (pscoldset==1) then
				psset = psset + Vpscoldset
			endif
        endif
! ---- Fault flag off

! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
                do is = 4,(nsv-1),2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
            if (nfile>0) then
! **** First call of time-step - read parameters from file and store
                if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
                endif
! **** Get parameters that were read form file and then stored
                propb    = saved_v(nsv+1)
                tint     = saved_v(nsv+2)
                tdt      = saved_v(nsv+3)
                dband    = saved_v(nsv+4)
                propbhl  = saved_v(nsv+5)
                pshlimit = saved_v(nsv+6)
                closloop = retolog(saved_v(nsv+7))
                sfspdman = saved_v(nsv+8)
            endif
! **** Run controller if mode is closed loop else use "manual" value
            if (closloop) then
! **** Closed loop mode
                if (time>saved_v(1)) then
! **** First call of timestep
! **** Update previous sample instant values
                    do is=4,(nsv-1),2
                        saved_v(is+1)=saved_v(is)
                    enddo
                endif
! **** Update previous values
                intp      = saved_v(5)
                difp      = saved_v(7)
                errp      = saved_v(9)
                pidoutp   = saved_v(11)
! **** Determine output of main control loop then apply over-ride if
! **** static pressure too high
! **** Apply deadband around set-point
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!Drexel added!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!              
! **** Pid controller
!                
                 pidout    = pidcont(ps,psset,propb,tint,tdt,intp,&
                                    difp,pidoutp,errp,tsamp,1.0,0.0)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!Drexel added!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! **** Determine whether auxiliary input switch is enabled or disabled
! **** disable auxiliary input if supply fan and return fan ok
                auxdis    = logicand(sfstatus,rfstatus)
! **** Determine controller output signal - select pidout if auxdis is
! **** true else select 0.0
                contout   = switch(auxdis,pidout,0.0)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!Drexel added!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                  sfspd    = contout

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!Drexel added!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! **** Save provisional values to be used in updating at next sample instant
                saved_v(4)  = intp
                saved_v(6)  = difp
                saved_v(8)  = errp
                saved_v(10) = pidout
! **** Open loop mode
            else
                sfspd     = sfspdman
            endif
! **** Save current sample number and output
            saved_v(2)  = float(nsample)
            saved_v(20) = sfspd
        else
! **** Not a sample instant, set output to value from previous sample
! **** instant
            sfspd = saved_v(20)
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = sfspd
! **** Disallow freezing
        do i=1,no
            iostat(i) = 0
        enddo
! **** Return
        return
        end subroutine type481

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     E51 return fan volume matching control
! *
! * PURPOSE:        Calculate return fan control signal from difference
! *                 between supply and return air volumetric flow rate.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. supflow : supply air volume flow rate sensor              (m3/s)
! *  2. retflow : return air volume flow rate sensor              (m3/s)
! *  3. sfstatus: supply fan status (1 = on, 0 = off)                (-)
! *  4. rfstatus: return fan status (1 = on, 0 = off)                (-)
! *  5. dflowset: setpoint for flow rate difference               (m3/s)
! *
! * OUTPUTS
! * =======
! *  1. rfspd   : return fan speed (0-1)                             (-)
! *
! * PARAMETERS
! * ==========
! *  1. propb   : proportional band                               (m3/s)
! *  2. tint    : integral time                                      (s)
! *  3. tdt     : derivative time                                    (s)
! *  4. deadb   : deadband                                        (m3/s)
! *  5. closloop: control mode (0 = open loop, 1 = closed loop)      (-)
! *  6. rfspdman: open loop return fan speed (0-1)                   (-)
! *  7. tsamp   : sampling interval                                  (s)
! *  8. nfile   : controller number (parameters in file contN.par)   (-)
! *
! * SAVED
! * ======
! *  1. time    : time of previous call
! *  2. nsample : sample number of previous controller execution
! *  3. nsample : sample number of previous controller sample
! *  4. intp    : integral term from previous call
! *  5. intp    : integral term from previous sample
! *  6. difp    : derivative term from previous call
! *  7. difp    : derivative term from previous sample
! *  8. errp    : error from previvous call
! *  9. errp    : error from previvous sample
! * 10. pidoutp : PID controller output from previous call
! * 11. pidoutp : PID controller output from previous sample
! * 12-17.      : (PAR(1)-PAR(6) read from file
! * 18. rfspd   : return fan speed from previous sample
! *********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 22, 1995
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:  DEADBAND
!   FUNCTIONS  CALLED:   RETOLOG,PIDCONT,LOGICAND,SWITCH,SPAN,
!                        LOGICAND
!
!   REVISION HISTORY:    Shokouh Pourarian, Jin Wen
!
!   REFERENCE:           ASHRAE 825-RP Final Report
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type482(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=11,ni=5,no=1,np=8,&
                                             nfp=6,ns=nsv+nfp+1
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

 ! **** Declaration of local variables
        real         :: intp
        logical      :: retolog,sfstatus,rfstatus,closloop,auxdis,&
                        logicand
        real         :: supflow,retflow,dflowset,propb,tint,&
                        tdt,deadb,rfspdman,tsamp,difp,errp,pidoutp,&
                        dflow,setdb,pidout,pidcont,rfspd,switch
        integer      :: nfile,is,nsample,i
        integer      :: itype=482

! **** Read in inputs
        supflow  = xin(1)
        retflow  = xin(2)
        sfstatus = retolog(xin(3))
        rfstatus = retolog(xin(4))
        dflowset = xin(5)
! **** Read in parameters
        propb    = par_v(1)
        tint     = par_v(2)
        tdt      = par_v(3)
        deadb    = par_v(4)
        closloop = retolog(par_v(5))
        rfspdman = par_v(6)
        tsamp    = par_v(7)
        nfile    = par_v(8)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
                do is = 4,(nsv-1),2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
            if (nfile>0) then
! **** First call of time-step - read parameters from file and store
               if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
               endif
! **** Get parameters that were read form file and then stored
               propb    = saved_v(nsv+1)
               tint     = saved_v(nsv+2)
               tdt      = saved_v(nsv+3)
               deadb    = saved_v(nsv+4)
               closloop = retolog(saved_v(nsv+5))
               rfspdman = saved_v(nsv+6)
            endif
! **** Run controller if mode is closed loop else use "manual" value
            if (closloop) then
! **** Closed loop mode
                if (time>saved_v(1)) then
! **** First call of timestep
! **** Update previous sample instant values
                    do is=4,(nsv-1),2
                        saved_v(is+1) = saved_v(is)
                    enddo
                endif
! **** Update previous values
                intp    = saved_v(5)
                difp    = saved_v(7)
                errp    = saved_v(9)
                pidoutp = saved_v(11)
! **** Difference of supply and return air volume
                dflow   = supflow - retflow

! **** Pid controller
                               
			    pidout  = pidcont(dflow,dflowset,propb,tint,tdt,intp,&
                                 difp,pidoutp,errp,tsamp,1.0,0.0)

! **** Determine whether auxiliary input switch is enabled or disabled
! **** disable auxiliary input if supply fan and return fan ok
                auxdis = logicand(sfstatus,rfstatus)
! **** Determine controller output signal - - select pidout if auxdis is
! **** true else select 0.0
                rfspd = switch(auxdis,pidout,0.0)
! **** Save provisional values to be used in updating at next sample instant
                saved_v(4)  = intp
                saved_v(6)  = difp
                saved_v(8)  = errp
                saved_v(10) = pidout
! **** Open loop mode
            else
                rfspd     = rfspdman
            endif
! **** Save current sample number and output value
            saved_v(2)  = float(nsample)
            saved_v(18) = rfspd
        else
! **** Not a sample instant, set output(s) to value(s) from previous sample
! **** instant
            rfspd = saved_v(18)
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = rfspd
! **** Disallow freezing
        do i=1,no
            iostat(i) = 0
        enddo
! **** Return
        return
        end subroutine type482

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     E-51 return fan reset control
! *
! * PURPOSE:        Determine set-point for return fan controller
! *
! ***********************************************************************
! * INPUTS
! * ======
! *  1. mindpr  : minimun outside air damper position (open=TRUE)     (-)
! *
! * OUTPUTS
! * =======
! *  1. dflowset: return fan controller set-point                  (m3/s)
! *
! * PARAMETERS
! * ==========
! *  1. dflowsop: return fan set-point when min OA damper open     (m3/s)
! *  2. tsamp   : sample time                                         (s)
! *  3. nfile   : controller number (parameters in file contN.par)    (-)
! *
! * SAVED
! * ======
! *  1. time    : time of previous call
! *  2. nsample : sample number of previous controller execution
! *  3. nsample : sample number of previous controller sample
! *  4.         : PAR(1) read from file
! *  5. dflowset: return fan controller set-point from previous call
! *
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! ************************************************************************

        subroutine type483(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=3,ni=1,no=1,np=3,&
                                             nfp=1,ns=nsv+nfp+1
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar
        integer                           :: itype=483

! **** Declaration of local variables
        logical      :: mindpr,retolog
        real         :: dflowsop,tsamp,dflowset,switch
        integer      :: nfile,nsample,i

! **** Read in inputs
        mindpr = retolog(xin(1))
! **** Read in parameters
        dflowsop = par_v(1)
        tsamp    = par_v(2)
        nfile    = par_v(3)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
            if (nfile>0) then
                if (time>saved_v(1)) then
! **** First call of timestep - read parameters from file and store
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    saved_v(nsv+1)=fpar(1)
                endif
                dflowsop  = saved_v(nsv+1)
             endif
! **** Run controller
! **** Set-point for return fan controller is zero if minimum outside air
! **** Damper is closed
            dflowset = switch(mindpr,dflowsop,0.0)
! **** Save current sample number and output value
            saved_v(2) = float(nsample)
            saved_v(5) = dflowset
        else
! **** Not a sample instant, set output(s) to value(s) from previous sample
! **** instant
            dflowset = saved_v(5)
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = dflowset
! **** Disallow freezing
        do i=1,no
            iostat(i) = 0
        enddo
! **** Return
	    return
	    end subroutine type483
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     E51 minimum outside air damper control
! *
! * PURPOSE:        Determine the demanded position of the minimun
! *                 outside air damper (fully open or fully closed)
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tsup    : supply air temperature sensor                      (C)
! *  2. tout    : outside air temperature sensor                     (C)
! *  3. sfstatus: supply fan status (1 = on, 0 = off)                (-)
! *
! * OUTPUTS
! * =======
! *  1. damprpos: minimun outside air damper position (0 or 1)       (-)
! *
! * PARAMETERS
! * ==========
! *  1. tsuplim : supply air temperature limit                       (C)
! *  2. toutlim : outside air temperature limit                      (C)
! *  3. tsamp   : sampling interval                                  (s)
! *  4. nfile   : controller number (parameters in file contN.par)   (-)
! *
! * SAVED
! * =====
! * 1.  time    : time of previous call of TYPE
! * 2.  nsample : sample number of previous controller execution
! * 3.  nsample : sample number of previous controller sample
! * 4-5.        : PAR(1)-PAR(2) read from file
! * 6.  damprpos: (controller output from previous sample)
! *
!**************************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   TSUPE:               November 22, 1995
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   RETOLOG,LOGICNOT,LOGICAND,LOGICOR,COMPARE
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           ASHRAE 825-RP Final Report
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! *************************************************************************

        subroutine type484(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=3,ni=3,no=1,np=4,&
                                             nfp=2,ns=nsv+nfp+1
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

! **** Declaration of local variables
        logical      :: retolog,sfstatus,l1,l2,l3,l4,l5,compare
        logical      :: logicand,logicor,logicnot
        real         :: tsup,tout,tsuplim,toutlim,tsamp,damprpos
        integer      :: nfile,nsample,i
        integer      :: itype=484

! **** Read in inputs
        tsup      = xin(1)
        tout      = xin(2)
        sfstatus = retolog(xin(3))
! **** Read in parameters
        tsuplim    = par_v(1)
        toutlim    = par_v(2)
        tsamp      = par_v(3)
        nfile      = par_v(4)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
		saved_v(1) = -999999.
		saved_v(2) = 0.
	    endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
            if (nfile>0) then
                if (time>saved_v(1)) then
! **** First call of timestep - read parameters from file and store
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    saved_v(nsv+1) = fpar(1)
                    saved_v(nsv+2) = fpar(2)
                endif
                tsuplim  = saved_v(nsv+1)
                toutlim  = saved_v(nsv+2)
             endif
! **** Run controller
! **** Compare outside air temperature with outside air temperature
! **** limit - true if toutlim > tout
            l1 = compare(toutlim,tout)
! **** Compare discharge air temperature with discharge air temperature
! **** limit - true if tsuplim > tsup
            l2 = compare(tsuplim,tsup)
! **** l1 nor l2
            l3 = logicor(l1,l2)
            l4 = logicnot(l3)
! **** Determine minimum outside air damper control signal
            l5 = logicand(sfstatus,l4)
            if (l5) then
! **** Damper is open if supply fan ok and tout > toutlim and tsup > tsuplim
                damprpos = 1.0
            else
                damprpos = 0.0
            endif
! **** Save current sample number and output
            saved_v(2) = float(nsample)
            saved_v(6) = damprpos
        else
! **** Not a sample instant, set output(s) to value(s) from previous sample
! **** instant
            damprpos = saved_v(6)
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = damprpos
! **** Disallow freezing (parameters read from file may change)
        do i=1,no
            iostat(i) = 0
        enddo
! **** Return
        return
        end subroutine type484
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     E51 modulated mixed air damper control
! *
! * PURPOSE:        Determines the mixed air damper control signal.
! *                 The damper positions are determined by a combination
! *                 of the supply air temperature controller (TYPE486)
! *                 and proportional control of the mixed air
! *                 temperature. If the supply fan status is not OK or
! *                 the outside enthalpy is higher than the return
! *                 enthalpy or low temperature over-ride is set,
! *                 the mixing box is set to full recirculation.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tmix    : mixed air temperature sensor                       (C)
! *  2. mbcdem  : mixing box cooling demand (0-1)                    (-)
! *  3. sfstatus: supply fan status (1 = on, 0 = off)                (-)
! *  4. lowtovr : low temperature over-ride (1 = low temperature)    (-)
! *  5. econ    : economizer status (1 = OA enthalpy > RA enthalpy)  (-)
! *
! * OUTPUTS
! * =======
! *  1. ddemo   : outside air damper demanded position (0-1)         (-)
! *  2. ddemr   : return air damper demanded position (0-1)          (-)
! *  3. ddeme   : exhaust air damper demanded position (0-1)         (-)
! *
! * PARAMETERS
! * ==========
! *  1. tset    : mixed air temperature setpoint                     (C)
! *  2. propb   : proportional band                                  (K)
! *  3. tint    : integral time                                      (s)
! *  4. tdt     : derivative time                                    (s)
! *  5. deadb   : deadband                                           (K)
! *  6. closloop: control mode (0 = open loop, 1 = closed loop)      (-)
! *  7. ddemmano: open loop outside air damper position (0-1)        (-)
! *  8. ddemmanr: open loop return air damper position (0-1)         (-)
! *  9. ddemmane: open loop exhaust air damper position (0-1)        (-)
! * 10. tsamp   : sampling interval                                  (s)
! * 11. nfile   : controller number (parameters in file contN.par)   (-)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call
! *  2. nsample : sample number of previous controller execution
! *  3. nsample : sample number of previous controller sample
! *  4. intp    : integral term from previous call
! *  5. intp    : integral term from previous sample
! *  6. difp    : derivative term from previous call
! *  7. difp    : derivative term from previous sample
! *  8. errp    : error from previous call
! *  9. errp    : error from previous sample
! * 10. pidoutp : output from previous call
! * 11. pidoutp : output from previous sample
! * 12-20.      : (PAR(1)-PAR(9) read from file
! * 21. ddemo   : Outside air damper demanded from previous call
! * 22. ddemr   : Return air damper demanded from previous call
! * 23. ddeme   : Exhaust air damper demanded from previous call
! **********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 22, 1995
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   RETOLOG,LOGICOR,LOGICNOT,LOGICAND,SWITCH,SPAN
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           ASHRAE 825-RP Final Report
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type485(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=11,ni=5,no=3,np=11,&
                                             nfp=9,ns=nsv+nfp+3
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

! **** Declaration of local variables
        real         :: mbcdem, intp           ! changed 12/6/1999
        logical      :: retolog,logicor,logicnot,logicand
        logical      :: sfstatus,lowtovr,econ,closloop,orloeo,norloeo,&
                        auxdis
        real         :: tmix,tset,propb,tint,tdt,deadb,ddemmano,&
                        ddemmanr,ddemmane,tsamp,difp,errp,pidoutp,&
                        tsetdb,pidout,pidcont,passthru,switch,ddemo,&
                        ddemr,ddeme
        integer      :: nfile,is,nsample,i
        integer      :: itype=485

! **** Read in inputs
        tmix        = xin(1)
        mbcdem      = xin(2)
        sfstatus    = retolog(xin(3))
        lowtovr     = retolog(xin(4))
        econ        = retolog(xin(5))
! **** Read in parameters
        tset        = par_v(1)
        propb       = par_v(2)
        tint        = par_v(3)
        tdt         = par_v(4)
        deadb       = par_v(5)
        closloop    = retolog(par_v(6))
        ddemmano    = par_v(7)
        ddemmanr    = par_v(8)
        ddemmane    = par_v(9)
        tsamp       = par_v(10)
        nfile       = par_v(11)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
                do is = 4,(nsv-1),2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Read parameters from file if controller number not zero
            if (nfile>0) then
! **** First call of time-step - read parameters from file and store
                if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
                endif
! **** Get parameters that were read form file and then stored
                tset     = saved_v(nsv+1)
                propb    = saved_v(nsv+2)
                tint     = saved_v(nsv+3)
                tdt      = saved_v(nsv+4)
                deadb    = saved_v(nsv+5)
                closloop = retolog(saved_v(nsv+6))
                ddemmano = saved_v(nsv+7)
                ddemmanr = saved_v(nsv+8)
                ddemmane = saved_v(nsv+9)
            endif
            if (closloop) then
! **** Closed loop mode
! **** First call of timestep - update previous sample instant values
                if (time>saved_v(1)) then
                    do is = 4,nsv-1,2
                        saved_v(is+1) = saved_v(is)
                    enddo
                endif
! **** Update previous values
                intp    = saved_v(5)
                difp    = saved_v(7)
                errp    = saved_v(9)
                pidoutp = saved_v(11)
! **** Apply deadband around set-point
                call deadband(tmix,tset,deadb,tsetdb)
! **** Pid controller
                pidout = pidcont(tmix,tsetdb,propb,tint,tdt,intp,difp,&
                                 pidoutp,errp,tsamp,2.0,0.0)
! **** Determine whether auxiliary input switch is enabled or disabled
! **** not low temperature over-ride or economizer
                orloeo = logicor(lowtovr,econ)
                norloeo = logicnot(orloeo)
! **** Auxiliary input disabled if supply fan status ok and neither
! **** low temperature over-ride or economizer is set
                auxdis = logicand(sfstatus,norloeo)
! **** Determine mixing box position demanded by supply air temperature
! **** controller - pass input through if
! **** auxilliary input disabled else set to zero (full recirculation)
                passthru = switch(auxdis,mbcdem,0.0)
! **** Determine demanded mixing box position - lesser of the mixed air
! **** temperature controller output and the (modoified) signal from the
! **** supply air temperature controller
                ddemo = min(pidout,passthru)
! **** Return and exhaust air dampers demanded position
                ddemr = 1.0 - ddemo
                ddeme = ddemo
! **** Save provisional values to be used in updating at next sample instant
                saved_v(4) = intp
                saved_v(6) = difp
                saved_v(8) = errp
                saved_v(10) = pidout
! **** Open loop mode
            else
                ddemo = ddemmano
                ddemr = ddemmanr
                ddeme = ddemmane
            endif
! **** Save current sample number
            saved_v(2)  = float(nsample)
! **** Save provisional values to be used in updating at next sample instant
            saved_v(21) = ddemo
            saved_v(22) = ddemr
            saved_v(23) = ddeme
        else
! ****Not a sample instant, set output to value from prev sample instant
            ddemo = saved_v(21)
            ddemr = saved_v(22)
            ddeme = saved_v(23)
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = ddemo
        yout(2) = ddemr
        yout(3) = ddeme
! **** Disallow freezing
        do i=1,no
            iostat(i) = 0
        enddo
! **** Return
        return
        end subroutine type485

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     ERS modulated mixed air damper control
! *
! * PURPOSE:        Determines the mixed air damper control signal.
! *                 The damper positions are determined by a combination
! *                 of the supply air temperature controller (TYPE486)
! *                 and proportional control of the mixed air
! *                 temperature. If the supply fan status is not OK or
! *                 the outside enthalpy is higher than the return
! *                 enthalpy or low temperature over-ride is set,
! *                 the mixing box is set to full recirculation.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tmix    : mixed air temperature sensor                       (C)
! *  2. mbcdem  : mixing box cooling demand (0-1)                    (-)
! *  3. sfstatus: supply fan status (1 = on, 0 = off)                (-)
! *  4. lowtovr : low temperature over-ride (1 = low temperature)    (-)
! *  5. econ    : economizer status (1 = OA enthalpy > RA enthalpy)  (-)
! *
! * OUTPUTS
! * =======
! *  1. ddemo   : outside air damper demanded position (0-1)         (-)
! *  2. ddemr   : return air damper demanded position (0-1)          (-)
! *  3. ddeme   : exhaust air damper demanded position (0-1)         (-)
! *
! * PARAMETERS
! * ==========
! *  1. tset    : mixed air temperature setpoint                     (C)
! *  2. propb   : proportional band                                  (K)
! *  3. tint    : integral time                                      (s)
! *  4. tdt     : derivative time                                    (s)
! *  5. deadb   : deadband                                           (K)
! *  6. closloop: control mode (0 = open loop, 1 = closed loop)      (-)
! *  7. ddemmano: open loop outside air damper position (0-1)        (-)
! *  8. ddemmanr: open loop return air damper position (0-1)         (-)
! *  9. ddemmane: open loop exhaust air damper position (0-1)        (-)
! * 10. oamin   : minimum outdoor air damper position                (-)
! * 11. tsamp   : sampling interval                                  (s)
! * 12. nfile   : controller number (parameters in file contN.par)   (-)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call
! *  2. nsample : sample number of previous controller execution
! *  3. nsample : sample number of previous controller sample
! *  4. intp    : integral term from previous call
! *  5. intp    : integral term from previous sample
! *  6. difp    : derivative term from previous call
! *  7. difp    : derivative term from previous sample
! *  8. errp    : error from previous call
! *  9. errp    : error from previous sample
! * 10. pidoutp : output from previous call
! * 11. pidoutp : output from previous sample
! * 12-20.      : (PAR(1)-PAR(9) read from file
! * 21. ddemo   : Outside air damper demanded from previous call
! * 22. ddemr   : Return air damper demanded from previous call
! * 23. ddeme   : Exhaust air damper demanded from previous call
! **********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 22, 1995
!
!   REVISED BY:          Shokouh Pourarian and Jin Wen
!                        Drexel University   
!
!   DATE:                May 27. 2014
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   RETOLOG,LOGICOR,LOGICNOT,LOGICAND,SWITCH,SPAN
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           ASHRAE 825-RP Final Report
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type585(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=11,ni=5,no=3,np=12,&
                                             nfp=9,ns=nsv+nfp+3
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

! **** Declaration of local variables
        real         :: mbcdem, intp           ! changed 12/6/1999
        logical      :: retolog,logicor,logicnot,logicand
        logical      :: sfstatus,lowtovr,econ,closloop,orloeo,norloeo,&
                        auxdis
        real         :: tmix,tset,propb,tint,tdt,deadb,ddemmano,&
                        ddemmanr,ddemmane,tsamp,difp,errp,pidoutp,&
                        tsetdb,pidout,pidcont,passthru,switch,ddemo,&
                        ddemr,ddeme,oamin
        integer      :: nfile,is,nsample,i
        integer      :: itype=585

! **** Read in inputs
        tmix        = xin(1)
        mbcdem      = xin(2)
        sfstatus    = retolog(xin(3))
        lowtovr     = retolog(xin(4))
        econ        = retolog(xin(5))
! **** Read in parameters
        tset        = par_v(1)
        propb       = par_v(2)
        tint        = par_v(3)
        tdt         = par_v(4)
        deadb       = par_v(5)
        closloop    = retolog(par_v(6))
        ddemmano    = par_v(7)
        ddemmanr    = par_v(8)
        ddemmane    = par_v(9)
		oamin       = par_v(10)
        tsamp       = par_v(11)
        nfile       = par_v(12)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
                do is = 4,(nsv-1),2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Read parameters from file if controller number not zero
            if (nfile>0) then
! **** First call of time-step - read parameters from file and store
                if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
                endif
! **** Get parameters that were read form file and then stored
                tset     = saved_v(nsv+1)
                propb    = saved_v(nsv+2)
                tint     = saved_v(nsv+3)
                tdt      = saved_v(nsv+4)
                deadb    = saved_v(nsv+5)
                closloop = retolog(saved_v(nsv+6))
                ddemmano = saved_v(nsv+7)
                ddemmanr = saved_v(nsv+8)
                ddemmane = saved_v(nsv+9)
            endif
            if (closloop) then
! **** Closed loop mode
! **** First call of timestep - update previous sample instant values
                if (time>saved_v(1)) then
                    do is = 4,nsv-1,2
                        saved_v(is+1) = saved_v(is)
                    enddo
                endif
! **** Update previous values
                intp    = saved_v(5)
                difp    = saved_v(7)
                errp    = saved_v(9)
                pidoutp = saved_v(11)
! **** Apply deadband around set-point
                call deadband(tmix,tset,deadb,tsetdb)
! **** Pid controller
!                pidout = pidcont(tmix,tsetdb,propb,tint,tdt,intp,difp,&
!                                 pidoutp,errp,tsamp,2.0,0.0)
! **** Drexel adds
                pidout=pidoutp
! **** Determine whether auxiliary input switch is enabled or disabled
! **** not low temperature over-ride or economizer
                orloeo = logicor(lowtovr,econ)
                norloeo = logicnot(orloeo)
! **** Auxiliary input disabled if supply fan status ok and neither
! **** low temperature over-ride or economizer is set
                auxdis = logicand(sfstatus,norloeo)
! **** Determine mixing box position demanded by supply air temperature
! **** controller - pass input through if
! **** auxilliary input disabled else set to zero (full recirculation)
                passthru = switch(auxdis,mbcdem,0.0)
! **** Determine demanded mixing box position - lesser of the mixed air
! **** temperature controller output and the (modoified) signal from the
! **** supply air temperature controller
!                ddemo = min(pidout,passthru)
! **** Drexel adds
                ddemo = max(mbcdem,oamin)
! ----  Fault flag on
		if (OAdamp==1) then
			ddemo = VOAdamp
!	        WRITE(*,*) 'OA damper is stuck'
		endif
! ----  Fault flag off
! **** Return and exhaust air dampers demanded position
                ddemr = 1.0 - ddemo
                ddeme = ddemo
! **** Save provisional values to be used in updating at next sample instant
                saved_v(4) = intp
                saved_v(6) = difp
                saved_v(8) = errp
                saved_v(10) = pidout
! **** Open loop mode
            else
                ddemo = ddemmano
                ddemr = ddemmanr
                ddeme = ddemmane
            endif
! **** Save current sample number
            saved_v(2)  = float(nsample)
! **** Save provisional values to be used in updating at next sample instant
            saved_v(21) = ddemo
            saved_v(22) = ddemr
            saved_v(23) = ddeme
        else
! ****Not a sample instant, set output to value from prev sample instant
            ddemo = saved_v(21)
            ddemr = saved_v(22)
            ddeme = saved_v(23)
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = ddemo
        yout(2) = ddemr
        yout(3) = ddeme
! **** Disallow freezing
        do i=1,no
            iostat(i) = 0
        enddo
! **** Return
        return
        end subroutine type585

!***********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!***********************************************************************
! * SUBROUTINE:    E51 supply air temperature control
! *
! * PURPOSE:       Calculate demanded position of cooling coil valve and
! *                mixing box dampers. Override at low temperatures or
! *                if supply fan status is not OK. At low temperatures,
! *                set cooling coil valve fully open (to prevent freezing).
! *                (The mixed air damper control (TYPE485) will set the
! *                mixing box to full recirc and the minimum outside air
! *                damper controller (TYPE484) will close the minimum
! *                outside air damper at low temperatures.)
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tsup    : supply air temperature sensor                      (C)
! *  2. sfstatus: supply fan status (1 = on, 0 = off)                (-)
! *  3. lowtovr : low temperature override signal (1=TRUE, 0=FALSE)  (-)
! *  4. tset    : supply air temperature setpoint                    (C)
! *
! * OUTPUTS
! * =======
! *  1. ddem    : damper cooling demand (0-1)                        (-)
! *  2. cdem    : cooling coil valve demand (0-1)                    (-)
! *
! * PARAMETERS
! * ==========
! *  1. propb   : proportional band                                  (K)
! *  2. tint    : integral time                                      (s)
! *  3. tdt     : derivative time                                    (s)
! *  4. cbreak  : breakpoint between damper and cooling coil demand(0-2)
! *  5. deadb   : deadband                                           (K)
! *  6. closloop: control mode (0=open loop, 1=closed loop)          (-)
! *  7. cdemman : open loop cooling coil demand (0-1)                (-)
! *  8. tsamp   : sampling interval                                  (s)
! *  9. nfile   : controller number (parameters in file contN.par)   (-)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call
! *  2. nsample : sample number of previous controller execution
! *  3. nsample : sample number of previous controller sample
! *  4. intp    : integral term from previous call
! *  5. intp    : integral term from previous sample
! *  6. difp    : derivative term from previous call
! *  7. difp    : derivative term from previous sample
! *  8. errp    : error from previous call
! *  9. errp    : error from previous sample
! * 10. pidoutp : output from previous call
! * 11. pidoutp : output from previous sample
! * 12-18.      : (PAR(1)-PAR(7) read from file
! * 19. ddem    : damper demand from previous sample
! * 20. cdem    : cooling coil demand from previous sample
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Based on the strategy used in Building E-51
!                        at MIT
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 22, 1995
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:  DEADBAND
!   FUNCTIONS  CALLED:   RETOLOG,PIDCONT,LOGICNOT,LOGICAND,SWITCH,SPAN
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           ASHRAE 825-RP Final Report
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type486(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=11,ni=4,no=2,np=9,&
                                             nfp=7,ns=nsv+nfp+2
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

! **** Declaration of local variables
        real         :: intp
        logical      :: retolog,logicnot,logicand
        logical      :: sfstatus,lowtovr,nlowtovr,auxdis,closloop
        real         :: tsup,tset,propb,tint,tdt,cbreak,deadb,cdemman,&
                        tsamp,difp,errp,pidoutp,tsetdb,pidout,pidcont,&
                        auxinp,switch,contout,ddem,span,cdem
        integer      :: nfile,is,nsample,i
        integer      :: itype=486

! **** Read in inputs
        tsup       = xin(1)
        sfstatus   = retolog(xin(2))
        lowtovr    = retolog(xin(3))
        tset       = xin(4)
! **** Read in parameters
        propb      = par_v(1)
        tint       = par_v(2)
        tdt        = par_v(3)
        cbreak     = par_v(4)
        deadb      = par_v(5)
        closloop   = retolog(par_v(6))
        cdemman    = par_v(7)
        tsamp      = par_v(8)
        nfile      = nint(par_v(9))
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
                do is = 4,nsv-1,2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
           if (nfile>0) then
! **** First call of time-step - read parameters from file and store
                if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
                endif
                propb      = saved_v(nsv+1)
                tint       = saved_v(nsv+2)
                tdt        = saved_v(nsv+3)
                cbreak     = saved_v(nsv+4)
                deadb      = saved_v(nsv+5)
                closloop   = retolog(saved_v(nsv+6))
                cdemman    = saved_v(nsv+7)
            endif
! **** Run controller if mode is closed loop else use "manual" value
            if (closloop) then
! **** Closed loop mode
                if (time>saved_v(1)) then
! **** First call of timestep
! **** Update previous sample instant values
                    do is=4,nsv-1,2
                        saved_v(is+1)=saved_v(is)
                    enddo
                endif
! **** Update previous values
                intp = saved_v(5)
                difp = saved_v(7)
                errp = saved_v(9)
                pidoutp = saved_v(11)
! **** Apply deadband around set-point
                call deadband(tsup,tset,deadb,tsetdb)
! **** Pid controller
                pidout = pidcont(tsup,tsetdb,propb,tint,tdt,intp,difp,&
                                 pidoutp,errp,tsamp,2.0,0.0)
! **** Determine whether auxiliary input switch is enabled or disabled
! **** compliment of low temperature over-ride
                nlowtovr = logicnot(lowtovr)
! **** Disable auxiliary input if fan ok and low temperature over-ride
! **** not set
                auxdis = logicand(sfstatus,nlowtovr)
! **** Determine auxiliary input value - cooling coil valve fully open
! **** if low temperature over-ride set else fully closed if fan status
! **** not ok and low temperature over-ride not set
                auxinp = switch(lowtovr,2.0,0.0)
! **** Determine controller output signal - select pidout if auxdis is
! **** true else select auxinp
                contout = switch(auxdis,pidout,auxinp)
! **** Sequence demands
! **** mixing damper demand
                if (cbreak/=0.0) then
                    ddem = span(contout,0.0,cbreak,0.0,1.0)
                else
                    stop 'type 486: cbreak = 0'
                endif
! **** Cooling coil demand
                if (cbreak/=2.0) then
                    cdem = span(contout,cbreak,2.0,0.0,1.0)
                else
                    stop 'type 486: cbreak = 2'
                endif
! **** Save provisional values to be used in updating at next sample instant
                saved_v(4) = intp
                saved_v(6) = difp
                saved_v(8) = errp
                saved_v(10) = pidout
! **** Open loop mode (dampers set manually in type265)
            else
                ddem = 0.0
                cdem = cdemman
            endif
! **** Save current sample number
            saved_v(2) = float(nsample)
! **** Save outputs for use when not a sample instant
            saved_v(19) = ddem
            saved_v(20) = cdem
        else
! **** Not a sample instant, set output to value from prev sample instant
            ddem = saved_v(19)
            cdem = saved_v(20)
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = ddem
        yout(2) = cdem
! **** Disallow freezing
        do i=1,no
            iostat(i) = 0
        enddo
! **** Return
        return
        end subroutine type486
!***********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!***********************************************************************
! * SUBROUTINE:    ERS dual duct system HD & CD supply air temperature 
! *                control
! *
! * PURPOSE:       Calculate demanded position of cooling  & heating coil 
! *                valves and mixing box dampers. Override at low temperatures 
! *                or if supply fans status is not OK. At low temperatures,
! *                set cooling coil valve fully open (to prevent freezing).
! *                (The mixed air damper control (TYPE485) will set the
! *                mixing box to full recirc and the minimum outside air
! *                damper controller (TYPE484) will close the minimum
! *                outside air damper at low temperatures.)
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tsupcd    : cold deck supply air temperature sensor            (C)
! *  2. tsuphd    : hot deck supply air temperature sensor             (C)
! *  3. sfstatuscd: cold deck supply fan status (1 = on, 0 = off)      (-)
! *  4. sfstatushd: hot deck supply fan status (1 = on, 0 = off)       (-)
! *  5. lowtovr   : low temperature override signal (1=TRUE, 0=FALSE)  (-)
! *  6. econ      : economizer statuse (1= OA temp<econtspt)           (-)
! *
! * OUTPUTS
! * =======
! *  1. ddem      : damper cooling demand (0-1)                        (-)
! *  2. cdem      : cooling coil valve demand (0-1)                    (-)
! *  3. hdem      : heating coil valve demand (0-1)                    (-)
! *
! * PARAMETERS
! * ==========
! *  1. propbcd   : cold deck proportional band                          (K)
! *  2. tintcd    : cold deck integral time                              (s)
! *  3. tdtcd     : cold deck derivative time                            (s)
! *  4. cbreak    : breakpoint between damper and cooling coil demand(0-2)
! *  5. deadbcd   : cold deck deadband                                   (K)
! *  6. closloopcd: cold deck control mode (0=open loop, 1=closed loop)  (-)
! *  7. tsetcd    : cold deck supply air temperature setpoint            (C)
! *  8. propbhd   : hot deck proportional band                          (K)
! *  9. tinthd    : hot deck integral time                              (s)
! * 10. tdthd     : hot deck derivative time                            (s)
! * 11. deadbhd   : hot deck deadband                                   (K)
! * 12. closloophd: hot deck control mode (0=open loop, 1=closed loop)  (-)
! * 13. tsethd    : hot deck supply air temperature setpoint             (C)
! * 14. cdemman   : open loop cooling coil demand (0-1)                  (-)
! * 15. hdemman   : open loop heating coil demand (0-1)                  (-)
! * 16. tsamp     : sampling interval                                    (s)
! * 17. nfile     : controller number (parameters in file contN.par)     (-)
! *
! * SAVED
! * =====
! *  1. time      : time of previous call
! *  2. nsample   : sample number of previous controller execution
! *  3. nsample   : sample number of previous controller sample
! *  4. intpcd    : cold deck integral term from previous call
! *  5. intpcd    : cold deck integral term from previous sample
! *  6. difpcd    : cold deck derivative term from previous call
! *  7. difpcd    : cold deck derivative term from previous sample
! *  8. errpcd    : cold deck error from previous call
! *  9. errpcd    : cold deck error from previous sample
! * 10. pidoutpcd : cold deck output from previous call
! * 11. pidoutpcd : cold deck output from previous sample
! * 12. intphd    : hot deck integral term from previous call
! * 13. intphd    : hot deck integral term from previous sample
! * 14. difphd    : hot deck derivative term from previous call
! * 15. difphd    : hot deck derivative term from previous sample
! * 16. errphd    : hot deck error from previous call
! * 17. errphd    : hot deck error from previous sample
! * 18. pidoutphd : hot deck output from previous call
! * 19. pidoutphd : hot deck output from previous sample
! * 20-34.        : (PAR(1)-PAR(15) read from file
! * 35. ddem      : damper demand from previous sample
! * 36. cdem      : cooling coil demand from previous sample
! * 37. hdem      : heating coil demand from previous sample
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Based on the strategy used in Building ERS at 
!                        Iowa Energy center
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 22, 1995
!
!   REVISED BY:          Shokouh Pourarian, Jin Wen
!                        Drexel University
!
!   DATE:                May 22, 2014
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:  DEADBAND
!   FUNCTIONS  CALLED:   RETOLOG,PIDCONT,LOGICNOT,LOGICAND,SWITCH,SPAN
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           ASHRAE 825-RP Final Report
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type586(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=19,ni=6,no=3,np=17,&
                                             nfp=15,ns=nsv+nfp+3
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

! **** Declaration of local variables
        real         :: intpcd,intphd
        logical      :: retolog,logicnot,logicand,logicor
        logical      :: sfstatuscd,sfstatushd,nsfstatuscd,nsfstatushd,lowtovr,nlowtovr,auxdiscd,auxdishd,closloopcd,closloophd,econ
        real         :: tsupcd,tsuphd,tsetcd,tsethd,propbcd,propbhd,tintcd,tinthd,tdtcd,tdthd,cbreak,deadbcd,deadbhd,cdemman,&
                        hdemman,tsamp,difpcd,difphd,errpcd,errphd,pidoutpcd,pidoutphd,tsetcddb,tsethddb,pidoutcd,pidouthd,pidcont,&
                        auxinp,switch,contout,ddem,span,cdem,hdem
        integer      :: nfile,is,nsample,i
        integer      :: itype=586

! **** Read in inputs
        tsupcd     = xin(1)
		tsuphd     = xin(2)
        sfstatuscd = retolog(xin(3))
		sfstatushd = retolog(xin(4))
        lowtovr    = retolog(xin(5))
        econ       = retolog(xin(6))
! **** Read in parameters
        propbcd    = par_v(1)
        tintcd     = par_v(2)
        tdtcd      = par_v(3)
        cbreak     = par_v(4)
        deadbcd    = par_v(5)
        closloopcd = retolog(par_v(6))
		tsetcd     = par_v(7)
		propbhd    = par_v(8)
        tinthd     = par_v(9)
        tdthd      = par_v(10)
        deadbhd    = par_v(11)
        closloophd = retolog(par_v(12))
		tsethd     = par_v(13)
        cdemman    = par_v(14)
		hdemman    = par_v(15)
        tsamp      = par_v(16)
        nfile      = nint(par_v(17))

! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
                do is = 4,nsv-1,2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
           if (nfile>0) then
! **** First call of time-step - read parameters from file and store
                if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
                endif
                propbcd    = saved_v(nsv+1)
                tintcd     = saved_v(nsv+2)
                tdtcd      = saved_v(nsv+3)
                cbreak     = saved_v(nsv+4)
                deadbcd    = saved_v(nsv+5)
                closloopcd = retolog(saved_v(nsv+6))
                tsetcd     = saved_v(nsv+7)
				propbhd    = saved_v(nsv+8)
                tinthd     = saved_v(nsv+9)
                tdthd      = saved_v(nsv+10)
                deadbhd    = saved_v(nsv+11)
                closloophd = retolog(saved_v(nsv+12))
                tsethd     = saved_v(nsv+13)
                cdemman    = saved_v(nsv+14)
				hdemman    = saved_v(nsv+15)
            endif
! **** Run controller if mode is closed loop else use "manual" value
            if (closloopcd.and.closloophd) then
! **** Closed loop mode
                if (time>saved_v(1)) then
! **** First call of timestep
! **** Update previous sample instant values
                    do is=4,nsv-1,2
                        saved_v(is+1)=saved_v(is)
                    enddo
                endif
! **** Update previous values
                intpcd    = saved_v(5)
                difpcd    = saved_v(7)
                errpcd    = saved_v(9)
                pidoutpcd = saved_v(11)
				
				intphd    = saved_v(13)
                difphd    = saved_v(15)
                errphd    = saved_v(17)
                pidoutphd = saved_v(19)
! **** Apply deadband around set-point
                call deadband(tsupcd,tsetcd,deadbcd,tsetcddb)
				call deadband(tsuphd,tsethd,deadbhd,tsethddb)
! **** Pid controller
! *******************************IF Econ is enabled***************************************
                pidoutcd = pidcont(tsupcd,tsetcddb,propbcd,tintcd,tdtcd,intpcd,difpcd,&
                                 pidoutpcd,errpcd,tsamp,2.0,0.0)

! *******************************IF Econ is DISabled**************************************
!                pidoutcd = pidcont(tsupcd,tsetcddb,propbcd,tintcd,tdtcd,intpcd,difpcd,&
!                                 pidoutpcd,errpcd,tsamp,1.0,0.0)
!
!				cdem     = pidoutcd
! *******************************IF Econ is DISabled**************************************
      
	            nsfstatuscd = logicnot(sfstatuscd)
	            nsfstatushd = logicnot(sfstatushd)
	  
	
! **** If HD & CD supply fans Off or low temperature over-ride AUXDIS is 'true',used for logic
	            auxdiscd = logicor(nsfstatuscd,lowtovr)
	            auxdishd = logicor(nsfstatushd,lowtovr)
	            if (auxdiscd.or.auxdishd) then 
	                ddem = 0.0
		            cdem = 0.0
		            hdem = 0.0
! *******************************IF Econ is enabled***************************************
	            elseif (pidoutcd>=1.0) then
	                if (econ) then
		                ddem = 0.0
		            else 
		                ddem = 1.0
		            endif
		            cdem = span (pidoutcd,1.0,2.0,0.0,1.0)
	            else
	                if (econ) then
		                ddem = 0.0
		            else 
		                ddem = pidoutcd
		            endif
		            cdem = 0.0
! *******************************IF Econ is enabled***************************************
	            endif 

				pidouthd = pidcont(tsuphd,tsethddb,propbhd,tinthd,tdthd,intphd,difphd,&
                                 pidoutphd,errphd,tsamp,1.0,0.0)
				hdem     = pidouthd
	      
! **** Save provisional values to be used in updating at next sample instant
                saved_v(4)  = intpcd
                saved_v(6)  = difpcd
                saved_v(8)  = errpcd
                saved_v(10) = pidoutcd

				saved_v(12) = intphd
                saved_v(14) = difphd
                saved_v(16) = errphd
                saved_v(18) = pidouthd
! **** Open loop mode (dampers set manually in type265)
            else
                ddem = 0.0
                cdem = cdemman
				hdem = hdemman
            endif
! **** Save current sample number
            saved_v(2) = float(nsample)
! **** Save outputs for use when not a sample instant
            saved_v(35) = ddem
            saved_v(36) = cdem
			saved_v(37) = hdem
        else
! **** Not a sample instant, set output to value from prev sample instant
            ddem = saved_v(35)
            cdem = saved_v(36)
			hdem = saved_v(37)
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = ddem
        yout(2) = cdem
		yout(3) = hdem
! **** Disallow freezing
        do i=1,no
            iostat(i) = 0
        enddo
! **** Return
        return
        end subroutine type586

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     E51 economizer control
! *
! * PURPOSE:        Determine the economizer control mode by comparing
! *                 outside and return air enthalpy
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tout    : outdoor air temperature sensor                     (C)
! *  2. rhout   : outdoor air relative humidity sensor (0-1)         (-)
! *  3. tret    : return air temperature sensor                      (C)
! *  4. rhret   : return air relative humidity sensor (0-1)          (-)
! *
! * OUTPUTS
! * =======
! *  1. econ    : economizer status (1 = OA enthalpy > RA enthalpy)  (-)
! *
! * PARAMETERS
! * ==========
! *  1. tsamp   : sampling interval                                  (s)
! *
! * SAVED
! * =====
! * 1.  time    : time of previous call of TYPE
! * 2.  nsample : sample number of previous controller execution
! * 3.  nsample : sample number of previous controller sample
! * 4.  econ    : controller output from previous sample
! *
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type487(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=3,ni=4,no=1,np=1,&
                                             nfp=0,ns=nsv+nfp+1
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

! **** Declaration of local variables
        real         :: logtore
        logical      :: econ,retolog,compare
        real         :: tout,rhout,tret,rhret,tsamp,oaenth,enthalpy,raenth
        integer      :: nsample,i

! **** Read in inputs
        tout = xin(1)
        rhout = xin(2)    ! changed 12/6/1999
        tret = xin(3)
        rhret = xin(4)    ! changed 12/6/1999
! **** Read in parameter
        tsamp = par_v(1)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
		saved_v(1) = -999999.
		saved_v(2) = 0.
	    endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Run controller
! **** Calculate enthalpy of outside air
            oaenth = enthalpy(tout,rhout)
! **** Calculate enthalpy of return air
            raenth = enthalpy(tret,rhret)
! **** Determine economizer control signal
            econ = compare(oaenth,raenth)
! **** Save current sample number and output
	    saved_v(2) = float(nsample)
	    saved_v(4) = logtore(econ)
        else
! **** Not a sample instant, set output(s) to value(s) from previous sample
! **** instant
            econ = retolog(saved_v(4))
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = logtore(econ)
! **** Allow freezing of algebraic variables
        do i=1,no
            iostat(i) = 1
        enddo
! **** Return
	    return
	    end subroutine type487
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     ERS economizer control
! * REVISED BY:     Shokouh Pourarian, Jin Wen
! *
! * PURPOSE:        Determine the economizer control mode by comparing
! *                 outside air temperature with economizer setpoint 
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tout    : outdoor air temperature sensor                     (C)
! *
! * OUTPUTS
! * =======
! *  1. econ    : economizer status (1 = OA Temp > EconTSPT)         (-)
!
! * PARAMETERS
! * ==========
! *  1. tsamp   : sampling interval                                  (s)
! *  2. econtspt: economizer air temperature setpoint (60F)          (C)
! *
! * SAVED
! * =====
! * 1.  time    : time of previous call of TYPE
! * 2.  nsample : sample number of previous controller execution
! * 3.  nsample : sample number of previous controller sample
! * 4.  econ    : controller output from previous sample
! *
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type587(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=3,ni=1,no=1,np=2,&
                                             nfp=0,ns=nsv+nfp+1
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

! **** Declaration of local variables
        real         :: logtore
        logical      :: econ,retolog,compare
        real         :: tout,tsamp,econtspt
        integer      :: nsample,i

! **** Read in inputs
        tout  = xin(1)
        
! **** Read in parameter
        tsamp    = par_v(1)
		econtspt = par_v(2)

! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
		saved_v(1) = -999999.
		saved_v(2) = 0.
	    endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
		    
			if (tout<econtspt) THEN
			    econ = .FALSE.
		    else 
			    econ = .TRUE.
		    endif

! **** Save current sample number and output
	    saved_v(2) = float(nsample)
	    saved_v(4) = logtore(econ)
        else
! **** Not a sample instant, set output(s) to value(s) from previous sample
! **** instant
            econ = retolog(saved_v(4))
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = logtore(econ)
! **** Allow freezing of algebraic variables
        do i=1,no
            iostat(i) = 1
        enddo
! **** Return
	    return
	    end subroutine type587

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     E51 low temperature override control
! *
! * PURPOSE:        Calculate low temperature override signal for
! *                 cooling coil and maximum outside air damper control
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tsup    : discharge air temperature                          (C)
! *  2. tout    : outdoor air temperature                            (C)
! *  3. sfstatus: supply fan status (1 = on, 0 = off)                (-)
! *
! * OUTPUTS
! * =======
! *  1. lowtovr : low temperature override signal                    (-)
! *
! * PARAMETERS
! * ==========
! *  1. tsuplim : supply air temperature limit                       (C)
! *  2. toutlim : outdoor air temperature limit                      (C)
! *  3. tsamp   : sample time                                        (s)
! *  4. nfile   : controller number (parameters in file contN.par)   (-)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call of TYPE
! *  2. nsample : sample number of previous controller execution
! *  3. nsample : sample number of previous controller sample
! *  4-5.       : PAR(1)-PAR(2) read from file
! *  6. lowtovr : controller output from previous sample
! *
!***********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   TSUPE:               November 22, 1995
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   RETOLOG,LOGICNOT,LOGICAND,LOGICOR,COMPARE,LOGTORE
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           ASHRAE 825-RP Final Report
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type488(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=3,ni=3,no=1,np=4,&
                                             nfp=2,ns=nsv+nfp+2
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar
        integer                           :: itype=488

! **** Declaration of local variables
        real         :: logtore
        logical      :: retolog,sfstatus,nsfstat,l1,l2,l3,l4,lowtovr,&
                        compare
        logical      :: logicnot,logicand,logicor
        real         :: tsup,tout,tsuplim,toutlim,tsamp
        integer      :: nfile,nsample,i

! **** Read in inputs
        tsup      = xin(1)
        tout      = xin(2)
        sfstatus  = retolog(xin(3))
! **** Read in parameters
        tsuplim   = par_v(1)
        toutlim   = par_v(2)
        tsamp     = par_v(3)
        nfile     = par_v(4)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
            if (nfile>0) then
                if (time>saved_v(1)) then
! **** First call of timestep - read parameters from file and store
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    saved_v(nsv+1)=fpar(1)
                    saved_v(nsv+2)=fpar(2)
                endif
                tsuplim   = saved_v(nsv+1)
                toutlim   = saved_v(nsv+2)
             endif
! **** Run controller
! **** Not supply fan status
            nsfstat = logicnot(sfstatus)
! **** Compare outside air temperature with outside air temperature
! **** limit - true if tout < toutlim
            l1 = compare(toutlim,tout)
! **** True if supply fan is not on and outside air temperature is lower
! **** than limit
            l2 = logicand(nsfstat,l1)
! **** Compare discharge air temperature with discharge air temperature
! **** limit - true if tsup < tsuplim
            l3 = compare(tsuplim,tsup)
! **** If supply fan is on and discharge air temperature is lower
! **** than limit
            l4 = logicand(sfstatus,l3)
! **** Determine low temperature over-ride - set if supply fan not ok and
! **** tout < toutlim, or if supply fan ok and tsup < tsuplim
            lowtovr = logicor(l2,l4)
! **** Save current sample number and output
            saved_v(2) = float(nsample)
            saved_v(6) = logtore(lowtovr)
        else
! **** Not a sample instant, set output(s) to value(s) from previous sample
! **** instant
            lowtovr = retolog(saved_v(6))
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = logtore(lowtovr)
! **** Disallow freezing (parameters read from file may change)
        do i=1,no
            iostat(i) = 0
        enddo
! **** Return
        return
        end subroutine type488

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     E51 supply air temperature reset
! *
! * PURPOSE:        Reset supply air temperature setpoint so as to keep
! *                 the maximum of several room temperatures at a
! *                 specified value.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tz(1)   : temperature in zone 1                              (C)
! *  2. tz(2)   : temperature in zone 2                              (C)
! *  3. tz(3)   : temperature in zone 3                              (C)
! *  4. tz(4)   : temperature in zone 4                              (C)
! *  5. tz(5)   : temperature in zone 5                              (C)
! *  6. tz(6)   : temperature in zone 6                              (C)
! *
! * OUTPUT
! * ======
! *  1. tsset   : supply air temperature setpoint                    (C)
! *
! * PARAMETERS
! * ==========
! *  1. tzset   : zone temperature setpoint                          (C)
! *  2. propb   : proportional band                                (K/K)
! *  3. tint    : integral time                                      (s)
! *  4. tssetmax: upper limit of the output                          (C)
! *  5. tssetmin: lower limit of the output                          (C)
! *  6. tsset0  : output at zero error (P), initial output (PI)      (C)
! *  7. deadb   : deadband                                           (K)
! *  8. ninputs : number of inputs                                   (-)
! *  9. tsamp   : sampling interval                                  (s)
! * 10. nfile   : controller number (parameters in file contN.par)   (s)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call
! *  2. nsample : sample number of previous controller execution
! *  3. nsample : sample number of previous controller sample
! *  4. intp    : integral term from previous call
! *  5. intp    : integral term from previous sample
! *  6. tssetp  : output from previous call
! *  7. tssetp  : output from previous sample
! *  8-14.      : (PAR(1)-PAR(7) read from file
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  No derivative action
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 22, 1995
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:  DEADBAND
!   FUNCTIONS  CALLED:   BIGGEST, PIDCONT
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           ASHRAE 825-RP Final Report
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type489(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=7,ni=6,no=1,np=10,&
                                             nfp=7,ns=nsv+nfp
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

! **** Declaration of local variables
        real,dimension(ni)                :: tz
        real         :: intp
        real         :: tzset,propb,tint,tssetmax,tssetmin,tsset0,&
                        deadb,tssetp,tzmax,biggest,tzsetdb,dtsmax,&
                        dtsmin,dummy,tsset,pidcont,tsamp
        integer      :: ninputs,nfile,i,nsample,is
        integer      :: itype=489

! **** Read in parameters
        tzset      = par_v(1)
        propb      = par_v(2)
        tint       = par_v(3)
        tssetmax   = par_v(4)
        tssetmin   = par_v(5)
        tsset0     = par_v(6)
        deadb      = par_v(7)
        ninputs    = nint(par_v(8))
        tsamp      = par_v(9)
        nfile      = par_v(10)
! **** Read in inputs
        do i=1,ninputs
            tz(i) = xin(i)
        enddo
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
! **** Initialize integral term and controller output
                saved_v(4) = 0.0
                saved_v(6) = tsset0
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample = nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
            if (nfile>0) then
! **** First call of time-step - read parameters from file and store
                if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
                endif
! **** Get parameters that were read form file and then stored
                tzset      = saved_v(nsv+1)
                propb      = saved_v(nsv+2)
                tint       = saved_v(nsv+3)
                tssetmax   = saved_v(nsv+4)
                tssetmin   = saved_v(nsv+5)
                tsset0     = saved_v(nsv+6)
                deadb      = saved_v(nsv+7)
            endif
            if (time>saved_v(1)) then
! **** First call of timestep
! **** Update previous sample instant values
                do is=4,(nsv-1),2
                    saved_v(is+1)=saved_v(is)
                enddo
            endif
! **** Update previous values
            intp   = saved_v(5)
            tssetp = saved_v(7)
! **** Select the highest room temperature
            tzmax = biggest(tz,ninputs)
! **** Apply dead-band
            call deadband(tzmax,tzset,deadb,tzsetdb)
! **** Pi control
! **** Setpoint can vary between tsset0+dtsmax and tsset0+dtsmin
            dtsmax = tssetmax-tsset0
            dtsmin = tssetmin-tsset0
! **** Pid output is offset of setpoint from tsset0
            tsset  = tsset0+pidcont(tzmax,tzsetdb,propb,tint,0.0,intp,&
                               dummy,tssetp,dummy,tsamp,dtsmax,dtsmin)
! **** Save current sample number
            saved_v(2) = float(nsample)
! **** Save provisional values to be used in updating at next sample instant
            saved_v(4) = intp
            saved_v(6) = tsset
        else
! **** Not a sample instant, set output(s) to value(s) from previous sample
! **** instant
            tsset = saved_v(6)
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = tsset
! **** Disallow freezing
        do i=1,no
            iostat(i) = 0
        enddo
! **** Return
        return
        end subroutine type489

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:    VAV room temperature control with reheat
! *
! * PURPOSE:       PI control of room temperature.
! *                Calculates the normalized demanded flow rate for a
! *                pressure-independent VAV box and the valve position
! *                for an associated reheat coil.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tzon    : space temperature sensor                           (C)
! *  2. tsup    : supply air temperature sensor                      (C)
! *  3. sfstatus: supply fan status (1 = on, 0 = off)                (-)
! *
! * OUTPUTS
! * =======
! *  1. vdem    : normalised volumetric flow demand (0-1)            (-)
! *  2. rdem    : reheat coil demand (0-1)                           (-)
! *  3. contout : room demand (heating and cooling) (-1 - +1)        (-)
! *
! * PARAMETERS
! * ==========
! *  1. tsetheat: heating setpoint for zone                          (C)
! *  2. tsetcool: cooling setpoint for zone                          (C)
! *  3. propb   : proportional band                                  (K)
! *  4. tint    : integral time                                      (s)
! *  5. tdt     : derivative time                                    (s)
! *  6. rbreak  : breakpoint btwn damper and reheat coil signals (-1-+1)
! *  7. tdmin   : minimum turndown ratio                             (-)
! *  8. closloop: control mode (0=open loop, 1=closed loop)          (-)
! *  9. vdemman : open loop demanded normalized volumetric flow rate (-)
! * 10. rdemman : open loop reheat coil demand (0-1)                 (-)
! * 11. tsamp   : sampling interval                                  (s)
! * 12. nfile   : controller number (parameters in file contN.par)   (-)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call
! *  2. nsample : sample number of previous controller execution
! *  3. nsample : sample number of previous controller sample
! *  4. intp    : integral term from previous call
! *  5. intp    : integral term from previous sample
! *  6. difp    : derivative term from previous call
! *  7. difp    : derivative term from previous sample
! *  8. errp    : error from previous call
! *  9. errp    : error from previous sample
! * 10. pidoutp : output from previous call
! * 11. pidoutp : output from previous sample
! * 12. suphot  : hysteresis output from previous call
! * 13. suphotp : hysteresis output from previous sample
! * 14-23.      : (PAR(1)-PAR(10) read from file
! * 24. vdem    : normalised volumetric flow set-point from previous call
! * 25. rdem    : reheat coil demand from previous call
! * 26. contout : room demand (heating and cooling) (-1 - +1)
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 22, 1995
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:  DEADBAND
!   FUNCTIONS  CALLED:   COMPHYS,RETOLOG,PIDCONT,LOGTORE,SWITCH,SPAN
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           ASHRAE 825-RP Final Report
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type490(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=13,ni=3,no=3,np=12,&
                                             nfp=10,ns=nsv+nfp+3
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

! **** Declaration of local variables
        real         :: logtore, intp
        logical      :: retolog,sfstatus,suphotp,suphot,comphys,&
                        closloop
        real         :: tzon,tsup,tsetheat,tsetcool,propb,tint,tdt,&
                        rbreak,tdmin,vdemman,rdemman,tsamp,contout,&
                        difp,errp,pidoutp,tset,deadb,tsetdb,pidout,&
                        pidcont,switch,vdem,span,rdem
        integer      :: nfile,is,nsample,i
        integer      :: itype=490

! **** Read in inputs
        tzon     = xin(1)           
        tsup     = xin(2)           
        sfstatus = retolog(xin(3))
! **** Read in parameters
        tsetheat = par_v(1)
        tsetcool = par_v(2)
        propb    = par_v(3)
        tint     = par_v(4)
        tdt      = par_v(5)
        rbreak   = par_v(6)
        tdmin    = par_v(7)
        closloop = retolog(par_v(8))
        vdemman  = par_v(9)
        rdemman  = par_v(10)
        tsamp    = par_v(11)
        nfile    = nint(par_v(12))
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
                contout = 0.0          ! 12/21/99
                do is = 4,nsv-1,2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
            if (nfile>0) then
! **** First call of time-step - read parameters from file and store
                if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
                endif
                tsetheat   = saved_v(nsv+1)
                tsetcool   = saved_v(nsv+2)
                propb      = saved_v(nsv+3)
                tint       = saved_v(nsv+4)
                tdt        = saved_v(nsv+5)
                rbreak     = saved_v(nsv+6)
                tdmin      = saved_v(nsv+7)
                closloop   = retolog(saved_v(nsv+8))
                vdemman    = saved_v(nsv+9)
                rdemman    = saved_v(nsv+10)
            endif
! **** Run controller if mode is closed loop else use "manual" value
            if (closloop) then
! **** Closed loop mode
                if (time>saved_v(1)) then
! **** First call of timestep
! **** Update previous sample instant values
                    do is=4,nsv-1,2
                        saved_v(is+1)=saved_v(is)
                    enddo
                endif
! **** Update previous values
                intp    = saved_v(5)
                difp    = saved_v(7)
                errp    = saved_v(9)
                pidoutp = saved_v(11)
                suphotp = retolog(saved_v(13))
! **** Apply deadband around set-point
                tset  = (tsetcool+tsetheat)/2.
                deadb = (tsetcool-tsetheat)/2.
                call deadband(tzon,tset,deadb,tsetdb)
! **** Pid controller - output in the range -1 - +1
                pidout = pidcont(tzon,tsetdb,propb,tint,tdt,intp,difp,&
                                 pidoutp,errp,tsamp,1.0,-1.0)
! **** Set output to zero if fan status not ok
                contout = switch(sfstatus,pidout,0.0)
! **** Sequence demands
! **** Vav damper demand - depends on whether supply air is hotter than
! **** Zone air - comparison uses 1 k hysteresis
                suphot = comphys(tsup,tzon,1.0,suphotp)
! **** Minimum turn-down ratio
                if (suphot) then
! **** Increased heating demand produces higher demanded flow rate
                    vdem = span(contout,-1.0,rbreak,tdmin,1.0) ! revised 12/6/99
                else
! **** Increased heating demand produces lower demanded flow rate
                    vdem = span(contout,rbreak,-1.0,tdmin,1.0) ! revised 12/6/99
                endif
! **** Reheat coil demand
                rdem = span(contout,rbreak,1.0,0.0,1.0)        ! revised 12/6/99
! **** Save provisional values to be used in updating at next sample instant
                saved_v(4)  = intp
                saved_v(6)  = difp
                saved_v(8)  = errp
                saved_v(10) = pidout
                saved_v(12) = logtore(suphot)
! **** Open loop mode - set manually
            else
                vdem = vdemman
                rdem = rdemman
            endif
! **** Save current sample number
            saved_v(2) = float(nsample)
! **** Save outputs for use when not a sample instant
            saved_v(24) = vdem
            saved_v(25) = rdem
            saved_v(26) = contout
        else
! **** Not a sample instant, set output to value from prev sample instant
            vdem    = saved_v(24)
            rdem    = saved_v(25)
            contout = saved_v(26)
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = vdem
        yout(2) = rdem
        yout(3) = contout
! **** Disallow freezing
        do i=1,no
            iostat(i) = 0
        enddo   
! **** Return
        return
        end  subroutine type490

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:    Dual duct VAV room temperature control 
! *
! * PURPOSE:       PI control of room temperature.
! *                Calculates the normalized demanded hot and cold flow rate 
! *                for a pressure-independent VAV dual duct box 
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. tzon      : space temperature sensor                                (C)
! *  2. sfstatuscd: cold deck supply fan status (1 = on, 0 = off)           (-)
! *  3. sfstatushd: hot deck supply fan status (1 = on, 0 = off)            (-)
! *
! * OUTPUTS
! * =======
! *  1. cvdem     : normalised volumetric cold flow demand (0-1)            (-)
! *  2. hvdem     : normalised volumetric hot flow demand (0-1)             (-)
! *
! * PARAMETERS
! * ==========
! *  1. tsetcool  : cooling setpoint for zone                               (C)
! *  2. tsetheat  : heating setpoint for zone                               (C)
! *  3. cvdemmin  : minimum normalised volumetric cold flow demand (0-1)    (-)
! *  4. hvdemmin  : minimum normalised volumetric hot flow demand (0-1)     (-)
! *  5. propb     : proportional band                                       (K)
! *  6. tint      : integral time                                           (s)
! *  7. tdt       : derivative time                                         (s)
! *  8. deadb     : deadband                                                (K)
! *  9. closloop  : control mode (0=open loop, 1=closed loop)               (-)
! * 10. cvdemman  : open loop demanded normalized volumetric cold flow rate (-)
! * 11. hvdemman  : open loop demanded normalized volumetric hot flow rate  (-)          
! * 12. tsamp     : sampling interval                                       (s)
! * 13. nfile     : controller number (parameters in file contN.par)        (-)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call
! *  2. nsample : sample number of previous controller execution
! *  3. nsample : sample number of previous controller sample
! *  4. intp    : integral term from previous call
! *  5. intp    : integral term from previous sample
! *  6. difp    : derivative term from previous call
! *  7. difp    : derivative term from previous sample
! *  8. errp    : error from previous call
! *  9. errp    : error from previous sample
! * 10. pidoutp : output from previous call
! * 11. pidoutp : output from previous sample
! * 12-22.      : (PAR(1)-PAR(11) read from file
! * 23. cvdem   : normalised volumetric cold flow set-point from previous call
! * 24. hvdem   : normalised volumetric hot flow set-point from previous call
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Shokouh Pourarian and Jin Wen
!                        Drexel University 
!
!   DATE:                May 28, 2014
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:  DEADBAND
!   FUNCTIONS  CALLED:   COMPHYS,RETOLOG,PIDCONT,LOGTORE,SWITCH,SPAN
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           
!
! **********************************************************************
!
!   Updated to Fortran 90 April 19, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type477(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=11,ni=3,no=2,np=13,&
                                             nfp=11,ns=nsv+nfp+2
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

! **** Declaration of local variables
        real         :: logtore, intp
        logical      :: retolog,sfstatuscd,sfstatushd,closloop
        real         :: tzon,tsetheat,tsetcool,cvdemmin,hvdemmin,propb,tint,tdt,&
                        deadb,cvdemman,hvdemman,tsamp,contout,cvdem,hvdem,&
                        difp,errp,pidoutp,tset,tsetheatdb,tsetcooldb,pidout,&
                        pidcont,switch
        integer      :: nfile,is,nsample,i,mode
        integer      :: itype=477

! **** Read in inputs
        tzon       = xin(1)                  
        sfstatuscd = retolog(xin(2))
		sfstatushd = retolog(xin(3))
! **** Read in parameters
        tsetcool = par_v(1)
        tsetheat = par_v(2)
        cvdemmin = par_v(3)
        hvdemmin = par_v(4)
        propb    = par_v(5)
        tint     = par_v(6)
        tdt      = par_v(7)
		deadb    = par_v(8)
        closloop = retolog(par_v(9))
        cvdemman = par_v(10)
        hvdemman = par_v(11)
        tsamp    = par_v(12)
        nfile    = nint(par_v(13))
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
			        saved_v(4) = 23.0
                do is = 6,nsv-1,2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
! **** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
! **** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
            if (nfile>0) then
! **** First call of time-step - read parameters from file and store
                if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
                endif
                tsetcool   = saved_v(nsv+1)
                tsetheat   = saved_v(nsv+2)
				cvdemmin   = saved_v(nsv+3)
                hvdemmin   = saved_v(nsv+4)
                propb      = saved_v(nsv+5)
                tint       = saved_v(nsv+6)
                tdt        = saved_v(nsv+7)
				deadb      = saved_v(nsv+8)
                closloop   = retolog(saved_v(nsv+9))
                cvdemman   = saved_v(nsv+10)
                hvdemman   = saved_v(nsv+11)

            endif
! **** Run controller if mode is closed loop else use "manual" value
            if (closloop) then
! **** Closed loop mode
                if (time>saved_v(1)) then
! **** First call of timestep
! **** Update previous sample instant values
                    do is=4,nsv-1,2
                        saved_v(is+1)=saved_v(is)
                    enddo
                endif
! **** Update previous values
                intp    = saved_v(5)  ! Tzone (previous)
                difp    = saved_v(7)
                errp    = saved_v(9)
                pidoutp = saved_v(11)
! **** Determine VAV dual duct box operation mode
!                If (tzon>= tsetcool.and.pidoutp<0.0) then
                If (tzon>= tsetcool) then
! **** Cooling mode
				    mode  = 1
!					if (mode .ne. difp) then
!					    intp    = 0.0
!						errp    = 0.0
!                        pidoutp = 0.0
!					endif
!                    elseif (mode == difp) then
!				    coolingmode  = 0
!					heatingmode  = 0 
!					satisfiedmode= 0
! **** Apply deadband around set-point
                    call deadband(tzon,tsetcool,deadb,tsetcooldb)
					tset         = tsetcooldb
! **** PID control loop calculates cooling demand
                    cvdemmin     = -cvdemmin
					pidout       = (1.0/propb)*((tzon-tset)*tsamp/tint + (tzon-intp))
!					pidout       = pidcont(tzon,tset,propb,tint,tdt,intp,difp,&
!                                   pidoutp,errp,tsamp,cvdemmin,-1.0)
					contout      = switch(sfstatuscd,pidout,0.0)
                    cvdem        = abs(contout)
					hvdem        = abs(hvdemmin)
!					endif
!				elseif (tzon<= tsetheat.and.pidoutp>0.0) then
                elseif (tzon<= tsetheat) then
! **** Heating mode
                    mode  = 2
!                    if (mode .ne. difp) then
!					    intp    = 0.0
!						errp    = 0.0
!                        pidoutp = 0.0
!					endif
!                    elseif (mode == difp) then
!				    coolingmode  = 0
!					heatingmode  = 1 
!					satisfiedmode= 0
! **** Apply deadband around set-point
                    call deadband(tzon,tsetheat,deadb,tsetheatdb)
					tset         = tsetheatdb
! **** PID control loop calculates heating demand
                    pidout       = (1.0*4.0/propb)*((tset-tzon)*tsamp/tint + (intp-tzon))                    
!					pidout       = pidcont(tzon,tset,propb,tint,tdt,intp,difp,&
!                                   pidoutp,errp,tsamp,1.0,hvdemmin)
					contout      = switch(sfstatushd,pidout,0.0)
                    hvdem        = abs(contout)
					cvdem        = abs(cvdemmin)
!					endif
				else
! **** Satisfied mode
                    mode  = 3
!                   coolingmode  = 0
!					heatingmode  = 1 
!					satisfiedmode= 0
					hvdem        = abs(hvdemmin)
					cvdem        = abs(cvdemmin)
!					intp         = 0.0
!                    difp         = 0.0
!                    errp         = 0.0
!                    pidout       = 0.0
				endif

! **** Save provisional values to be used in updating at next sample instant
                saved_v(4)  = tzon
!                saved_v(6)  = difp
                saved_v(6)  = mode
                saved_v(8)  = errp
                saved_v(10) = pidout
! **** Open loop mode - set manually
            else
                cvdem = cvdemman
                hvdem = hvdemman
            endif
! **** Save current sample number
            saved_v(2) = float(nsample)
! **** Save outputs for use when not a sample instant
            saved_v(20) = cvdem
            saved_v(21) = hvdem
            
        else
! **** Not a sample instant, set output to value from prev sample instant
            cvdem    = saved_v(20)
            hvdem    = saved_v(21)
            
        endif
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = cvdem
        yout(2) = hvdem
        
! **** Disallow freezing
        do i=1,no
            iostat(i) = 0
        enddo   
! **** Return
        return
        end  subroutine type477
!***********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!***********************************************************************
! * SUBROUTINE:    Fan coil unit supply air temperature control                
! *
! * LANGUAGE:      FORTRAN 77
! *
! * PURPOSE:       Calculate demanded position of cooling coil valve, 
! *                heating coil valve and outdoor air damper for fan 
! *                unit based on ERS control strategy.
! *                PID1: controls cooling coil valve
! *                PID2: controls heating coil valve
! *                PID3: controls outdoor air damper position
! *                
! **********************************************************************
! * INPUTS
! * ======
! *  1. FCUMODE : fan coil mode (1= occupied, 0=unoccupied)          (-)
! *  2. FANMODE : fan mode (1= auto, 2=cycle, 3=on)                  (-)
! *  3. TZONE   : space air temperature sensor                       (C)
! *  4. TMA     : mixed air temperature sensor                       (C)
! *
! * OUTPUTS
! * =======
! *  1. DDEM    : damper demand (0-1)                                (-)
! *  2. CDEM    : cooling coil valve demand (0-1)                    (-)
! *  3. HDEM    : heating coil valve demand (0-1)                    (-)
! *  4. FANSP   : fan speed (0=off, 1=low, 2=medium, 3=high)         (-)
! *  5. N       : rotational speed of the fan                    (rev/s)
! *
! * PARAMETERS
! * ==========
! *  1. TSETHEAT: heating setpoint for zone                          (C)
! *  2. TSETCOOL: cooling setpoint for zone                          (C)
! *  3. TSETMIXA: mixed air temperature setpoint                     (C)
! *  4. FCUOAMIN: outdoor air minimum damper position (0-1)          (-)
! *  5. FANSPMAN: fan speed in cycle or on mode 
! *               (0=off, 1=low, 2=medium, 3=high)                   (-)
! *  6. NLOW    : fan rotational speed in low speed              (rev/s)
! *  7. NMED    : fan rotational speed in medium speed           (rev/s)
! *  8. NHIGH   : fan rotational speed in high speed             (rev/s)
! *  9. PROPB1  : proportional band for PID1                         (K)
! * 10. TINT1   : integral time for PID1                             (s)
! * 11. TDT1    : derivative time for PID1                           (s)
! * 12. PROPB2  : proportional band for PID2                         (K)
! * 13. TINT2   : integral time for PID2                             (s)
! * 14. TDT2    : derivative time for PID2                           (s)
! * 15. PROPB3  : proportional band for PID3                         (K)
! * 16. TINT3   : integral time for PID3                             (s)
! * 17. TDT3    : derivative time for PID3                           (s)
! * 18. CLOSLOOP: control mode (0=open loop, 1=closed loop)          (-)
! * 19. FCUOAMAN: open loop outdoor air damper position (0-1)        (-)
! * 20. CDEMMAN : open loop cooling coil valve demand (0-1)          (-)
! * 21. HDEMMAN : open loop heating coil valve demand (0-1)          (-)
! * 22. TSAMP   : sampling interval                                  (s)
! * 23. NFILE   : controller number (parameters in file contN.par)   (-)
! *
! * SAVED
! * =====
! *  1. TIME     : time of previous call
! *  2. NSAMPLE  : sample number of previous controller execution
! *  3. NSAMPLE  : sample number of previous controller sample
! *  4. INTP1    : integral term from previous call of PID1
! *  5. INTP1    : integral term from previous sample of PID1
! *  6. DIFP1    : derivative term from previous call of PID1
! *  7. DIFP1    : derivative term from previous sample of PID1
! *  8. ERRP1    : error from previous call of PID1
! *  9. ERRP1    : error from previous sample of PID1
! * 10. PIDOUT1P : PID1 output from previous call of PID1
! * 11. PIDOUT1P : PID1 output from previous sample of PID1
! * 12. INTP2    : integral term from previous call of PID2
! * 13. INTP2    : integral term from previous sample of PID2
! * 14. DIFP2    : derivative term from previous call of PID2
! * 15. DIFP2    : derivative term from previous sample of PID2
! * 16. ERRP2    : error from previous call of PID2
! * 17. ERRP2    : error from previous sample of PID2
! * 18. PIDOUT2P : PID2 output from previous call of PID2
! * 19. PIDOUT2P : PID2 output from previous sample of PID2
! * 20. INTP3    : integral term from previous call of PID3
! * 21. INTP3    : integral term from previous sample of PID3
! * 22. DIFP3    : derivative term from previous call of PID3
! * 23. DIFP3    : derivative term from previous sample of PID3
! * 24. ERRP3    : error from previous call of PID3
! * 25. ERRP3    : error from previous sample of PID3
! * 26. PIDOUT3P : PID3 output from previous call of PID3
! * 27. PIDOUT3P : PID3 output from previous sample of PID3
! * 28-48.       : (PAR(1)-PAR(21) read from file
! * 49. DDEM     : damper demand from previous sample
! * 50. CDEM     : cooling coil demand from previous sample
! * 51. HDEM     : heating coil demand from previous sample
! * 52. FANSP    : fan speed from previous sample
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Based on the strategy used in ERS
!                        
! * DEVELOPER:           Shokouh Pourarian and Jin Wen
! *                      Drexel University
! *
! * LAST MODIFIED:       May, 2012
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:  
!   FUNCTIONS  CALLED:   RETOLOG,PIDCONT
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           
!
! **********************************************************************
!
        subroutine type491(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=27,ni=4,no=5,np=23,&
                                             nfp=21,ns=nsv+nfp+4
	    real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

! **** Declaration of local variables
        real         :: INTP1,INTP2,INTP3,N,NLOW,NMED,NHIGH
		integer      :: FANMODE,FANSP,CLGMODE,HTGMODE,FANSPMAN
	    logical      :: RETOLOG,CLOSLOOP,FCUMODE
        real         :: TZONE,TMA,DDEM,CDEM,HDEM,TSETHEAT,TSETCOOL,&
		                TSETMIXA,FCUOAMIN,PROPB1,TINT1,TDT1,PROPB2,&
						TINT2,TDT2,PROPB3,TINT3,TDT3,FCUOAMAN,CDEMMAN,&
                        HDEMMAN,TSAMP,DIFP1,ERRP1,PIDOUT1P,DIFP2,ERRP2,& 
						PIDOUT2P,DIFP3,ERRP3,PIDOUT3P,PIDOUT1,PIDOUT2,&
						PIDOUT3,PIDCONT
        integer      :: nfile,is,nsample,i
        integer      :: itype=491	  
	  
!     flag system (Shun on)***************************************************** 
!      COMMON    /controller/ CONmixdamp,CONheat,CONcool,CONsequence,
!     &                       CONreverse,CONrf,CONsf
!	COMMON    /Vcontroller/VCONmixdamp,VCONheat,VCONcool,VCONsequence, 
!     &                       VCONreverse,VCONrf,VCONsf
!     Controller category:
!     mixdamp - mixing dampers; 
!     heat - heating coil;  cool - cooling coil; 
!     sequence - sequence of heating and cooling devices;
!	reverse - reverse action; SF - Supply Fan; RF - Return Fan;
!     /controller/* : 0 - fault free; 1 - unstable
!     /Vcontroller/  VCON*f  : stick at a fixed speed
!     flag system (Shun off)**************************************************** 

!*** Read in inputs
        FCUMODE    = retolog(xin(1))
	    FANMODE    = nint(xin(2))
        TZONE      = xin(3)
        TMA        = xin(4)
!*** Read in parameters
        TSETHEAT   = par_v(1)
        TSETCOOL   = par_v(2)
	    TSETMIXA   = par_v(3)
	    FCUOAMIN   = par_v(4)
	    FANSPMAN   = par_v(5)
        NLOW       = par_v(6)
        NMED       = par_v(7)
	    NHIGH      = par_v(8)
	    PROPB1     = par_v(9)
        TINT1      = par_v(10)
        TDT1       = par_v(11)
	    PROPB2     = par_v(12)
        TINT2      = par_v(13)
        TDT2       = par_v(14)
	    PROPB3     = par_v(15)
        TINT3      = par_v(16)
        TDT3       = par_v(17)
        CLOSLOOP   = retolog(par_v(18))
	    FCUOAMAN   = par_v(19)
        CDEMMAN    = par_v(20)
	    HDEMMAN    = par_v(21)      
        TSAMP      = par_v(22)
        NFILE      = nint(par_v(23))

!*** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
!                FANSP=0.0
                do is = 4,nsv-1,2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
!*** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
!*** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
            if (nfile>0) then
! **** First call of time-step - read parameters from file and store
                if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
                endif
                  
			    TSETHEAT   = saved_v(nsv+1)
	            TSETCOOL   = saved_v(nsv+2)
	            TSETMIXA   = saved_v(nsv+3)
	            FCUOAMIN   = saved_v(nsv+4)
			    FANSPMAN   = saved_v(nsv+5)
                NLOW       = saved_v(nsv+6)
                NMED       = saved_v(nsv+7)
                NHIGH      = saved_v(nsv+8)
	            PROPB1     = saved_v(nsv+9)
                TINT1      = saved_v(nsv+10)
	            TDT1       = saved_v(nsv+11)
			    PROPB2     = saved_v(nsv+12)
                TINT2      = saved_v(nsv+13)
	            TDT2       = saved_v(nsv+14)
			    PROPB3     = saved_v(nsv+15)
                TINT3      = saved_v(nsv+16)
	            TDT3       = saved_v(nsv+17)   
                CLOSLOOP   = retolog(saved_v(nsv+18))
	            FCUOAMAN   = saved_v(nsv+19)
                CDEMMAN    = saved_v(nsv+20)
			    HDEMMAN    = saved_v(nsv+21)                
		    endif
!*** Run controller if mode is closed loop else use "manual" value
            IF (CLOSLOOP) THEN
!*** Closed loop mode
                IF (time>saved_v(1)) THEN
!*** First call of timestep
!*** Update previous sample instant values
                    do is=4,nsv-1,2
                        saved_v(is+1)=saved_v(is)
                    enddo
                ENDIF
!*** Update previous values
                INTP1    = saved_v(5)
                DIFP1    = saved_v(7)
                ERRP1    = saved_v(9)
                PIDOUT1P = saved_v(11)
	            INTP2    = saved_v(13)
                DIFP2    = saved_v(15)
                ERRP2    = saved_v(17)
                PIDOUT2P = saved_v(19)
	            INTP3    = saved_v(21)
                DIFP3    = saved_v(23)
                ERRP3    = saved_v(25)
                PIDOUT3P = saved_v(27)
	            FANSP    = saved_v(52)
!    Fault flag on***********************************************
!			  IF (CONmixdamp==1) THEN
!				  PROPB = 100/VCONmixdamp
!			  ENDIF
!			  IF (CONheat==1) THEN
!				  PROPB = 100/VCONheat
!			  ENDIF
!			  IF (CONcool==1) THEN
!				  PROPB = 100/VCONcool
!			  ENDIF
!    Fault flag off***********************************************

!*** If fan coil unit is in occupied mode cooling/heating coil valves and outdoor air damper is adjusted
!*** Fan coil unit is in occupied mode
	          IF (FCUMODE) THEN
!*** Determine the mode of fan coil unit		     
	              IF (TZONE>=(TSETCOOL-1)) THEN 
				      CLGMODE=1
	                  HTGMODE=0
	          	  ELSEIF (TZONE<=(TSETHEAT+1))THEN
	                  CLGMODE=0
	                  HTGMODE=1
	              ELSE 
				      CLGMODE=0
	                  HTGMODE=0
	              ENDIF
!*** zone calls for cooling					  
				  IF (CLGMODE==1)THEN 
!*** PID1 controller for cooling valve and fan speed in fan auto mode and PID3 controller for outdoor air damper position  	                  
				      
	                  PIDOUT1 = PIDCONT(TZONE,TSETCOOL,PROPB1,TINT1,TDT1,&
					                    INTP1,DIFP1,PIDOUT1P,ERRP1,TSAMP,0.0,-1.0)
	                  PIDOUT1 = ABS(PIDOUT1)
	                  PIDOUT3 = PIDCONT(TMA,TSETMIXA,PROPB3,TINT3,TDT3,&
					                    INTP3,DIFP3,PIDOUT3P,ERRP3,TSAMP,0.0,-1.0)
	                  PIDOUT3 = ABS(PIDOUT3)

	                  DDEM  = PIDOUT3
	                  CDEM  = PIDOUT1
	                  HDEM  = 0.0
!*** fan is in auto mode
				      IF (FANMODE==1) THEN       
	                       IF (PIDOUT1>0.AND.PIDOUT1<0.35)THEN
	                           FANSP = 1.0
	                       ELSEIF(PIDOUT1>0.45.AND.PIDOUT1<0.75)THEN
                               FANSP = 2.0
                           ELSEIF(PIDOUT1>0.85.AND.PIDOUT1<=1.0)THEN
                               FANSP = 3.0
	                       ELSEIF(PIDOUT1>=0.35.AND.PIDOUT1<=0.45)THEN
	                             IF (FANSP==0.0.OR.FANSP==1.0)THEN
                                     FANSP = 1.0
	                             ELSE
	                                 FANSP = 2.0                             
								 ENDIF
	                       ELSEIF(PIDOUT1>=0.75.AND.PIDOUT1<=0.85)THEN
	                             IF (FANSP==3.0)THEN
                                     FANSP = 3.0
                                 ELSE
                                     FANSP = 2.0
                                 ENDIF
	                       ENDIF         
					  ENDIF
!*** fan is in cycle or on mode					       
					  IF (FANMODE==2.OR.FANMODE==3) THEN  
	                      FANSP = FANSPMAN
					  ENDIF
!*** zone calls for heating					      
				  ELSEIF (HTGMODE==1) THEN  

!*** PID2 controller for heating valve 
                      
	                  PIDOUT2 = PIDCONT(TZONE,TSETHEAT,PROPB2,TINT2,TDT2,&
                                        INTP2,DIFP2,PIDOUT2P,ERRP2,TSAMP,1.0,0.0)
                                           
	                  DDEM  = FCUOAMIN
	                  CDEM  = 0.0
	                  HDEM  = PIDOUT2
!*** fan is in auto mode
				      IF (FANMODE==1) THEN       
	                       IF (PIDOUT2>0.AND.PIDOUT2<0.35)THEN
                               FANSP = 1.0
	                       ELSEIF(PIDOUT2>0.45.AND.PIDOUT2<0.75)THEN
	                           FANSP = 2.0
                             ELSEIF(PIDOUT2>0.85.AND.PIDOUT2<=1.0)THEN
                               FANSP = 3.0
	                       ELSEIF(PIDOUT2>=0.35.AND.PIDOUT2<=0.45)THEN
                               IF (FANSP==0.0.OR.FANSP==1.0)THEN
                                   FANSP = 1.0
	                           ELSE
	                               FANSP = 2.0                             
							   ENDIF
	                       ELSEIF(PIDOUT2>=0.75.AND.PIDOUT2<=0.85)THEN
                               IF (FANSP==3.0)THEN
	                               FANSP = 3.0
                               ELSE
                                   FANSP = 2.0
							   ENDIF
	                       ENDIF         
					  ENDIF
!*** fan is in cycle or on mode     
					  IF (FANMODE==2.OR.FANMODE==3) THEN  
                          FANSP = FANSPMAN
                      ENDIF
!*** fan coil is in economizer mode					                                     
                    ELSE
!*** in economizer mode fan is off in Auto & cycle mode				                                    
                        IF (FANMODE==1.OR.FANMODE==2) THEN 
					      DDEM  = FCUOAMIN
	                      CDEM  = 0.0
	                      HDEM  = 0.0
	                      FANSP = 0.0
!*** in economizer mode fan speed is equal to preset value in on mode
                        ELSEIF (FANMODE==3.0) THEN                
	                      DDEM  = FCUOAMIN
	                      CDEM  = 0.0
	                      HDEM  = 0.0
	                      FANSP = FANSPMAN
	                    ENDIF                                       
                    ENDIF
!*** Fan coil unit is in unoccupied mode				                                  
              ELSE       
                  DDEM  = 0.0
	              CDEM  = 0.0
	              HDEM  = 0.0
	              FANSP = 0.0
	          ENDIF      
	           
!*** Save provisional values to be used in updating at next sample instant
                saved_v(4)  = INTP1
                saved_v(6)  = DIFP1
                saved_v(8)  = ERRP1
                saved_v(10) = PIDOUT1
	            saved_v(12) = INTP2
                saved_v(14) = DIFP2
                saved_v(16) = ERRP2
                saved_v(18) = PIDOUT2
	            saved_v(20) = INTP3
                saved_v(22) = DIFP3
                saved_v(24) = ERRP3
                saved_v(26) = PIDOUT3
!*** Open loop mode (dampers set manually in TYPE265)
            ELSE
                DDEM = FCUOAMAN
                CDEM = CDEMMAN
	            HDEM = HDEMMAN
			    FANSP= FANSPMAN   
            ENDIF
!*** Save current sample number
          saved_v(2) = FLOAT(NSAMPLE)
!*** Save outputs for use when not a sample instant        
		  saved_v(49) = DDEM
          saved_v(50) = CDEM
		  saved_v(51) = HDEM
	      saved_v(52) = FANSP
        ELSE
!*** Not a sample instant, set output to value from prev sample instant            
		  DDEM = saved_v(49)
          CDEM = saved_v(50)
	      HDEM = saved_v(51)
	      FANSP= saved_v(52)
        ENDIF
!*** Save time of current call
      saved_v(1) = TIME
!*** Output
        
	  IF (FANSP==0)THEN
	      N=0.0
	  ELSEIF (FANSP==1.0)THEN
	      N=NLOW 
	  ELSEIF (FANSP==2.0)THEN
	      N=NMED
	  ELSEIF (FANSP==3.0)THEN
	      N=NHIGH
	  ENDIF
	  yout(1) = DDEM
      yout(2) = CDEM
      yout(3) = HDEM
	  yout(4) = FANSP
	  yout(5) = N	  	  	    
! **** Disallow freezing
      do i=1,no
         iostat(i) = 0
      enddo   
! **** Return
      return
      end  subroutine type491
	  

!***********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!***********************************************************************
! * SUBROUTINE:    Fan coil unit supply air temperature control                
! *
! * LANGUAGE:      FORTRAN 77
! *
! * PURPOSE:       Calculate demanded position of cooling coil valve, 
! *                heating coil valve and outdoor air damper for fan 
! *                unit based on ERS control strategy.
! *                PID1: controls cooling coil valve
! *                PID2: controls heating coil valve
! *                PID3: controls outdoor air damper position
! *                
! **********************************************************************
! * INPUTS
! * ======
! *  1. FCUMODE : fan coil mode (1= occupied, 0=unoccupied)              (-)
! *  2. FANMODE : fan mode (1= auto, 2=cycle, 3=on)                      (-)
! *  3. TZONE   : space air temperature sensor                           (C)
! *  4. TMA     : mixed air temperature sensor                           (C)
! *
! * OUTPUTS
! * =======
! *  1. DDEM       : damper demand (0-1)                                 (-)
! *  2. CDEM       : cooling coil valve demand (0-1)                     (-)
! *  3. HDEM       : heating coil valve demand (0-1)                     (-)
! *  4. N          : rotational speed of the fan                     (rev/s)
! *  5. THERMALMODE: fan coil unit thermal mode
! *                  (0=unoccupied, 1= cooling, 2= heating, 3=economizer)(-)
! *
! * PARAMETERS
! * ==========
! *  1. TSETHEAT: heating setpoint for zone                          (C)
! *  2. TSETCOOL: cooling setpoint for zone                          (C)
! *  3. TSETMIXA: mixed air temperature setpoint                     (C)
! *  4. FCUOAMIN: outdoor air minimum damper position (0-1)          (-)
! *  5. FANSPMAN: fan speed in cycle or on mode 
! *               (0=off, 1=low, 2=medium, 3=high)                   (-)
! *  6. NLOW    : fan rotational speed in low speed              (rev/s)
! *  7. NMED    : fan rotational speed in medium speed           (rev/s)
! *  8. NHIGH   : fan rotational speed in high speed             (rev/s)
! *  9. PROPB1  : proportional band for PID1                         (K)
! * 10. TINT1   : integral time for PID1                             (s)
! * 11. TDT1    : derivative time for PID1                           (s)
! * 12. PROPB2  : proportional band for PID2                         (K)
! * 13. TINT2   : integral time for PID2                             (s)
! * 14. TDT2    : derivative time for PID2                           (s)
! * 15. PROPB3  : proportional band for PID3                         (K)
! * 16. TINT3   : integral time for PID3                             (s)
! * 17. TDT3    : derivative time for PID3                           (s)
! * 18. CLOSLOOP: control mode (0=open loop, 1=closed loop)          (-)
! * 19. FCUOAMAN: open loop outdoor air damper position (0-1)        (-)
! * 20. CDEMMAN : open loop cooling coil valve demand (0-1)          (-)
! * 21. HDEMMAN : open loop heating coil valve demand (0-1)          (-)
! * 22. TSAMP   : sampling interval                                  (s)
! * 23. NFILE   : controller number (parameters in file contN.par)   (-)
! *
! * SAVED
! * =====
! *  1. TIME     : time of previous call
! *  2. NSAMPLE  : sample number of previous controller execution
! *  3. NSAMPLE  : sample number of previous controller sample
! *  4. INTP1    : integral term from previous call of PID1
! *  5. INTP1    : integral term from previous sample of PID1
! *  6. DIFP1    : derivative term from previous call of PID1
! *  7. DIFP1    : derivative term from previous sample of PID1
! *  8. ERRP1    : error from previous call of PID1
! *  9. ERRP1    : error from previous sample of PID1
! * 10. PIDOUT1P : PID1 output from previous call of PID1
! * 11. PIDOUT1P : PID1 output from previous sample of PID1
! * 12. INTP2    : integral term from previous call of PID2
! * 13. INTP2    : integral term from previous sample of PID2
! * 14. DIFP2    : derivative term from previous call of PID2
! * 15. DIFP2    : derivative term from previous sample of PID2
! * 16. ERRP2    : error from previous call of PID2
! * 17. ERRP2    : error from previous sample of PID2
! * 18. PIDOUT2P : PID2 output from previous call of PID2
! * 19. PIDOUT2P : PID2 output from previous sample of PID2
! * 20. INTP3    : integral term from previous call of PID3
! * 21. INTP3    : integral term from previous sample of PID3
! * 22. DIFP3    : derivative term from previous call of PID3
! * 23. DIFP3    : derivative term from previous sample of PID3
! * 24. ERRP3    : error from previous call of PID3
! * 25. ERRP3    : error from previous sample of PID3
! * 26. PIDOUT3P : PID3 output from previous call of PID3
! * 27. PIDOUT3P : PID3 output from previous sample of PID3
! * 28-48.       : (PAR(1)-PAR(21) read from file
! * 49. DDEM     : damper demand from previous sample
! * 50. CDEM     : cooling coil demand from previous sample
! * 51. HDEM     : heating coil demand from previous sample
! * 52. FANSP    : fan speed from previous sample
! * 53. N        : fan rotational speed from previous sample
! * 54. THERMALMODE : FCU thermal mode from previous sample
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Based on the strategy used in ERS
!                        
! * DEVELOPER:           Shokouh Pourarian and Jin Wen
! *                      Drexel University
! *
! * LAST MODIFIED:       May, 2012
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:  
!   FUNCTIONS  CALLED:   RETOLOG,PIDCONT
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           
!
! **********************************************************************
!
        subroutine type479(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: nsv=27,ni=4,no=5,np=23,&
                                             nfp=21,ns=nsv+nfp+6
	    real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
        real,dimension(nfp)               :: fpar

! **** Declaration of local variables
        real         :: INTP1,INTP2,INTP3,N,NLOW,NMED,NHIGH
		integer      :: FANMODE,CLGMODE,HTGMODE,FANSPMAN
		integer      :: THERMALMODE, FANSP 
	    logical      :: RETOLOG,CLOSLOOP,FCUMODE
        real         :: TZONE,TMA,DDEM,CDEM,HDEM,TSETHEAT,TSETCOOL,&
		                TSETMIXA,FCUOAMIN,PROPB1,TINT1,TDT1,PROPB2,&
						TINT2,TDT2,PROPB3,TINT3,TDT3,FCUOAMAN,CDEMMAN,&
                        HDEMMAN,TSAMP,DIFP1,ERRP1,PIDOUT1P,DIFP2,ERRP2,& 
						PIDOUT2P,DIFP3,ERRP3,PIDOUT3P,PIDOUT1,PIDOUT2,&
						PIDOUT3,PIDCONT
        integer      :: nfile,is,nsample,i
        integer      :: itype=479	  
	  

!*** Read in inputs
        FCUMODE    = retolog(xin(1))
	    FANMODE    = nint(xin(2))
        TZONE      = xin(3)
        TMA        = xin(4)
!*** Read in parameters
        TSETHEAT   = par_v(1)
        TSETCOOL   = par_v(2)
	    TSETMIXA   = par_v(3)
	    FCUOAMIN   = par_v(4)
	    FANSPMAN   = par_v(5)
        NLOW       = par_v(6)
        NMED       = par_v(7)
	    NHIGH      = par_v(8)
	    PROPB1     = par_v(9)
        TINT1      = par_v(10)
        TDT1       = par_v(11)
	    PROPB2     = par_v(12)
        TINT2      = par_v(13)
        TDT2       = par_v(14)
	    PROPB3     = par_v(15)
        TINT3      = par_v(16)
        TDT3       = par_v(17)
        CLOSLOOP   = retolog(par_v(18))
	    FCUOAMAN   = par_v(19)
        CDEMMAN    = par_v(20)
	    HDEMMAN    = par_v(21)      
        TSAMP      = par_v(22)
        NFILE      = nint(par_v(23))

!*** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
                saved_v(2) = 0.
            endif
            if (init==0) then
!                FANSP=0.0
                do is = 4,nsv-1,2
                    saved_v(is) = 0.0
                enddo
            endif
        endif
!*** Update number of previous sample if new step-time
        if (time>saved_v(1)) then
            saved_v(3) = saved_v(2)
        endif
!*** Run controller if a sample instant
        nsample=nint(saved_v(3))+1
        if (time>=(nsample*tsamp) .or. time==0.0) then
! **** Use parameters from file if controller number not zero
            if (nfile>0) then
! **** First call of time-step - read parameters from file and store
                if (time>saved_v(1)) then
                    call rfile(nfile,'cont',nfp,itype,fpar)
                    do i=1,nfp
                        saved_v(nsv+i)=fpar(i)
                    enddo
                endif
                  
			    TSETHEAT   = saved_v(nsv+1)
	            TSETCOOL   = saved_v(nsv+2)
	            TSETMIXA   = saved_v(nsv+3)
	            FCUOAMIN   = saved_v(nsv+4)
			    FANSPMAN   = saved_v(nsv+5)
                NLOW       = saved_v(nsv+6)
                NMED       = saved_v(nsv+7)
                NHIGH      = saved_v(nsv+8)
	            PROPB1     = saved_v(nsv+9)
                TINT1      = saved_v(nsv+10)
	            TDT1       = saved_v(nsv+11)
			    PROPB2     = saved_v(nsv+12)
                TINT2      = saved_v(nsv+13)
	            TDT2       = saved_v(nsv+14)
			    PROPB3     = saved_v(nsv+15)
                TINT3      = saved_v(nsv+16)
	            TDT3       = saved_v(nsv+17)   
                CLOSLOOP   = retolog(saved_v(nsv+18))
	            FCUOAMAN   = saved_v(nsv+19)
                CDEMMAN    = saved_v(nsv+20)
			    HDEMMAN    = saved_v(nsv+21)                
		    endif
!*** Run controller if mode is closed loop else use "manual" value
            IF (CLOSLOOP) THEN
!*** Closed loop mode
                IF (time>saved_v(1)) THEN
!*** First call of timestep
!*** Update previous sample instant values
                    do is=4,nsv-1,2
                        saved_v(is+1)=saved_v(is)
                    enddo
                ENDIF
!*** Update previous values
                INTP1    = saved_v(5)
                DIFP1    = saved_v(7)
                ERRP1    = saved_v(9)
                PIDOUT1P = saved_v(11)
	            INTP2    = saved_v(13)
                DIFP2    = saved_v(15)
                ERRP2    = saved_v(17)
                PIDOUT2P = saved_v(19)
	            INTP3    = saved_v(21)
                DIFP3    = saved_v(23)
                ERRP3    = saved_v(25)
                PIDOUT3P = saved_v(27)
	            FANSP    = saved_v(52)
!    Fault flag on***********************************************
!			  IF (CONmixdamp==1) THEN
!				  PROPB = 100/VCONmixdamp
!			  ENDIF
!			  IF (CONheat==1) THEN
!				  PROPB = 100/VCONheat
!			  ENDIF
!			  IF (CONcool==1) THEN
!				  PROPB = 100/VCONcool
!			  ENDIF
!    Fault flag off***********************************************

!*** If fan coil unit is in occupied mode cooling/heating coil valves and outdoor air damper is adjusted
!*** Fan coil unit is in occupied mode
	          IF (FCUMODE) THEN
!*** Determine the mode of fan coil unit		     
	              IF (TZONE>=(TSETCOOL-0.5556)) THEN 
				      CLGMODE     = 1
	                  HTGMODE     = 0
					  THERMALMODE = 1
	          	  ELSEIF (TZONE<=(TSETHEAT+0.5556))THEN
	                  CLGMODE     = 0
	                  HTGMODE     = 1
					  THERMALMODE = 2
	              ELSE 
				      CLGMODE=0
	                  HTGMODE=0
					  THERMALMODE = 3
	              ENDIF
!*** zone calls for cooling					  
				  IF (CLGMODE==1)THEN 
!*** PID1 controller for cooling valve and fan speed in fan auto mode and PID3 controller for outdoor air damper position  	                  
				      
	                  PIDOUT1 = PIDCONT(TZONE,TSETCOOL,PROPB1,TINT1,TDT1,&
					                    INTP1,DIFP1,PIDOUT1P,ERRP1,TSAMP,0.0,-1.0)
					  
	                  PIDOUT1 = ABS(PIDOUT1)
!*** Fault flag on******************************************************************
!                      if (CONcoolreverse==0)then
!					      PIDOUT1 = PIDOUT1
!					  elseif (CONcoolreverse==1)then
!					      PIDOUT1 = 1.0 - PIDOUT1
!					  endif
!*** Fault flag off******************************************************************
!	                  PIDOUT3 = PIDCONT(TMA,TSETMIXA,PROPB3,TINT3,TDT3,&
!					                    INTP3,DIFP3,PIDOUT3P,ERRP3,TSAMP,0.3 ,-1.0)
!	                  PIDOUT3 = ABS(PIDOUT3)
					  PIDOUT3 = FCUOAMIN

					      DDEM    = PIDOUT3
	                      CDEM    = PIDOUT1
	                      HDEM    = 0.0
					      INTP2   = 0.0   
					      DIFP2   = 0.0   
					      ERRP2   = 0.0  
					      PIDOUT2P= 0.0               
!*** fan is in auto mode
				      IF (FANMODE==1) THEN       
	                       IF (PIDOUT1>0.AND.PIDOUT1<0.35)THEN
	                           FANSP = 1.0
	                       ELSEIF(PIDOUT1>0.45.AND.PIDOUT1<0.75)THEN
                               FANSP = 2.0
                           ELSEIF(PIDOUT1>0.85.AND.PIDOUT1<=1.0)THEN
                               FANSP = 3.0
	                       ELSEIF(PIDOUT1>=0.35.AND.PIDOUT1<=0.45)THEN
	                             IF (FANSP==0.0.OR.FANSP==1.0)THEN
                                     FANSP = 1.0
	                             ELSE
	                                 FANSP = 2.0                             
								 ENDIF
	                       ELSEIF(PIDOUT1>=0.75.AND.PIDOUT1<=0.85)THEN
	                             IF (FANSP==3.0)THEN
                                     FANSP = 3.0
                                 ELSE
                                     FANSP = 2.0
                                 ENDIF
	                       ENDIF         
					  ENDIF
!*** fan is in cycle or on mode					       
					  IF (FANMODE==2.OR.FANMODE==3) THEN  
	                      FANSP = FANSPMAN
					  ENDIF
!*** zone calls for heating					      
				  ELSEIF (HTGMODE==1) THEN  

!*** PID2 controller for heating valve 
                      
	                  PIDOUT2 = PIDCONT(TZONE,TSETHEAT,PROPB2,TINT2,TDT2,&
                                        INTP2,DIFP2,PIDOUT2P,ERRP2,TSAMP,0.0,-1.0)
					  PIDOUT2 = ABS(PIDOUT2)
!*** Fault flag on******************************************************************
!                      if (CONheatreverse==0)then
!					      PIDOUT2 = PIDOUT2
!					  elseif (CONheatreverse==1)then
!					      PIDOUT2 = 1.0 - PIDOUT2
!					  endif
!*** Fault flag off******************************************************************                                           

	                  DDEM    = FCUOAMIN
	                  CDEM    = 0.0
	                  HDEM    = PIDOUT2
					  INTP1   = 0.0   
					  DIFP1   = 0.0   
					  ERRP1   = 0.0  
					  PIDOUT1P= 0.0 
					  INTP3   = -FCUOAMIN   
					  DIFP3   = 0.0   
					  ERRP3   = 0.0  
					  PIDOUT3P= FCUOAMIN
!*** fan is in auto mode
				      IF (FANMODE==1) THEN       
	                       IF (PIDOUT2>0.AND.PIDOUT2<0.35)THEN
                               FANSP = 1.0
	                       ELSEIF(PIDOUT2>0.45.AND.PIDOUT2<0.75)THEN
	                           FANSP = 2.0
                             ELSEIF(PIDOUT2>0.85.AND.PIDOUT2<=1.0)THEN
                               FANSP = 3.0
	                       ELSEIF(PIDOUT2>=0.35.AND.PIDOUT2<=0.45)THEN
                               IF (FANSP==0.0.OR.FANSP==1.0)THEN
                                   FANSP = 1.0
	                           ELSE
	                               FANSP = 2.0                             
							   ENDIF
	                       ELSEIF(PIDOUT2>=0.75.AND.PIDOUT2<=0.85)THEN
                               IF (FANSP==3.0)THEN
	                               FANSP = 3.0
                               ELSE
                                   FANSP = 2.0
							   ENDIF
	                       ENDIF         
					  ENDIF
!*** fan is in cycle or on mode     
					  IF (FANMODE==2.OR.FANMODE==3) THEN  
                          FANSP = FANSPMAN
                      ENDIF
!*** fan coil is in economizer mode					                                     
                  ELSE
					  DDEM    = FCUOAMIN
	                  CDEM    = 0.0
	                  HDEM    = 0.0
					  INTP1   = 0.0   
					  DIFP1   = 0.0   
					  ERRP1   = 0.0  
					  PIDOUT1P= 0.0
					  INTP2   = 0.0   
					  DIFP2   = 0.0   
					  ERRP2   = 0.0  
					  PIDOUT2P= 0.0 
					  INTP3   = -FCUOAMIN   
					  DIFP3   = 0.0   
					  ERRP3   = 0.0  
					  PIDOUT3P= FCUOAMIN
!*** in economizer mode fan is off in Auto & cycle mode				                                    
                      IF (FANMODE==1.OR.FANMODE==2) THEN 
					      FANSP   = 0.0
						  
!*** in economizer mode fan speed is equal to preset value in on mode
                      ELSEIF (FANMODE==3.0) THEN                
	                      FANSP = FANSPMAN
	                    
					  ENDIF                                       
                    ENDIF
!*** Fan coil unit is in unoccupied mode				                                  
              ELSE       
                  DDEM        = 0.0
	              CDEM        = 0.0
	              HDEM        = 0.0
	              FANSP       = 0.0
				  THERMALMODE = 0.0
				  INTP1       = 0.0   
				  DIFP1       = 0.0
				  ERRP1       = 0.0
				  PIDOUT1P    = 0.0
				  INTP2       = 0.0
				  DIFP2       = 0.0
				  ERRP2       = 0.0
				  PIDOUT2P    = 0.0
				  INTP3       = 0.0   
				  DIFP3       = 0.0   
				  ERRP3       = 0.0  
			      PIDOUT3P    = 0.0
	          ENDIF       
!*** Save provisional values to be used in updating at next sample instant
              saved_v(4)  = INTP1
              saved_v(6)  = DIFP1
              saved_v(8)  = ERRP1
              saved_v(10) = CDEM
	          saved_v(12) = INTP2
              saved_v(14) = DIFP2
              saved_v(16) = ERRP2
              saved_v(18) = HDEM
	          saved_v(20) = INTP3
              saved_v(22) = DIFP3
              saved_v(24) = ERRP3
              saved_v(26) = DDEM
			  
!*** Open loop mode (dampers set manually)
            ELSE
                DDEM = FCUOAMAN
                CDEM = CDEMMAN
	            HDEM = HDEMMAN
			    FANSP= FANSPMAN
				IF (CDEMMAN.NE.0.0) THEN 
				    THERMALMODE = 1
				ELSEIF (HDEMMAN.NE.0.0) THEN
				    THERMALMODE = 2
				ELSE 
				    THERMALMODE = 3
				ENDIF   
            ENDIF
		  
		  IF (FANSP==0)THEN
	          N=0.0
	      ELSEIF (FANSP==1.0)THEN
	          N=NLOW 
	      ELSEIF (FANSP==2.0)THEN
	          N=NMED
	      ELSEIF (FANSP==3.0)THEN
	          N=NHIGH
	      ENDIF
!*** Save current sample number
          saved_v(2) = FLOAT(NSAMPLE)
!*** Save outputs for use when not a sample instant        
		  saved_v(49) = DDEM
          saved_v(50) = CDEM
		  saved_v(51) = HDEM
	      saved_v(52) = FANSP
		  saved_v(53) = N
		  saved_v(54) = THERMALMODE

        ELSE
!*** Not a sample instant, set output to value from prev sample instant            
		  DDEM        = saved_v(49)
          CDEM        = saved_v(50)
	      HDEM        = saved_v(51)
	      N           = saved_v(53)
          THERMALMODE = saved_v(54)
        ENDIF
!*** Save time of current call
      saved_v(1) = TIME
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!      IF (time.eq.32400)THEN
!	      CDEM=0.52344
!		  saved_v(4)  = -0.52189
!		  saved_v(6)  = 0.0
!          saved_v(8)  = -0.0694
!          saved_v(10) = 0.52344
!          HDEM=0.623130
!		  saved_v(12) = -0.59081
!          saved_v(14) = 0.0
!          saved_v(16) = 1.736
!          saved_v(18) = -0.623130
!	   ENDIF
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!	      
!*** Output
      yout(1) = DDEM
      yout(2) = CDEM
      yout(3) = HDEM
	  yout(4) = N
	  yout(5) = THERMALMODE	  	  	    
! **** Disallow freezing
      do i=1,no
         iostat(i) = 0
      enddo   
! **** Return
      return
      end  subroutine type479 

	    
